/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CGrid.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CGrid.
 *   
 *  @author Christoph Brzozowski.
 *  
 */

#ifndef cls_CGridH
#define cls_CGridH

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

#include <stdlib.h>
#include <memory.h>
#include <malloc.h>

#include "CAbstractGrid.h"

// *******************************************************************
//
// Klasse: CGrid
//
// Erweitert: CAbstractGrid
//
// Zweck : Klassentemplate fr ein zweidimensionales dynamisches 
//			   Array. Durch die Implementierung als Template kann das
//         Grid Werte eines beliebigen Typs T aufnehmen. 
//
// Hinweis:
//    
//         ACHTUNG!!! Die Klasse ist NICHT geeignet zur Aufnahme von
//         Zeiger-/Referenztypen, da beim Lschen oder nder der Gre
//				 des Grids die freigewordenen Elemente nicht freigeben werden
//				 knnen. Hierzu empfiehlt es sich eine weitere Klasse 
//         abzuleiten, die die Methode DisposeElement berschreibt,
//         und das Flag DisposeData im Konstruktor auf TRUE setzt.
//
// Autor:  Christoph Brzozowski
//
// *******************************************************************

/*! 
 * 
 * @class CGrid
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Zweidimensionales dynamisches Array beliebigen Elementtyps.
 *
 *	Das Klassentemplate implementiert auf der Basis der \b CAbstractGrid -
 *	Klasse ein zweidimensionales dynamisches Array. Der Elementtyp wird
 *	durch den Templateparameter \b T festgelegt.
 *  Da der Speicher fr die Elemente dynamisch alloziiert wird, knnen die
 *  Ausmae des Arrays zur Laufzeit gendert werden.
 *
 *  \note
 *  Die Klasse eignet sich nicht zur Aufbewahrung von Referenzen oder dynamisch
 *  alloziierten Daten, da bei einer Grennderung des Arrays, insbesondere
 *  einer Verkleinerung, der Speicher fr die berschssigen Elemente nicht
 *  freigegeben wird. 
 *  Die Klasse verfgt jedoch bereits ber Mechanismen, die eine
 *  Implementierung eines solchen Verhaltens in abgeleiteten Klassen
 *  erleichtern.
 */

template <class T> class CGrid : public CAbstractGrid
{
	
protected:

	/*! \brief Freigabeflag fr Elementdaten.
	 *
	 *  Mit Hilfe dieses Flags kann festgelegt werden, ob bei einer
	 *  Grennderung des Arrays, oder beim Lschen der Elemente
	 *  der Speicher, der durch diese Elemente verbraucht wird
	 *  freigegeben werden soll. Dies ist nur sinnvoll, falls es sich
	 *  bei den Elementen um Objektreferenzen oder Zeiger handelt.
	 *
	 */
	bool DisposeData;  
										 
	//! Gibt den Speicher frei, den ein bestimmtes Element belegt.

	/*
	 *  Es macht nur sinn diese Methode aufzurufen, falls es sich beim dem 
	 *  Elementyp \b T um eine Objektreferenz, oder einen Zeiger auf dynamisch
	 *  alloziierte Daten handelt. Durch einen Aufruf, wird mittels \b delete()
	 *  der Speicher freigegeben bzw. das Objekt, auf das die Referenz zeigt, zerstrt. 
	 *  Der Zeiger wird anschlieend auf \b NULL gesetzt.
	 *
	 */
	virtual void disposeElement(int intX,int intY);

public:

	//! Standardkonstruktor.
	CGrid();

	//! Zusatzkonstruktor.
	CGrid(int intWidth, int intHeight);

	//! Kopierkonstruktor.
	CGrid(CGrid<T>& objGrid);

	//! ndert die Ausmae des Grids.
	virtual void resize(int intNewWidth, int intNewHeight);

	//! Lscht den Inhalt des Grids.
	virtual void clear();
	
	//! Lscht den Inhalt des Grids mit dem Element \b objClearElement.
	virtual void clear(const T& objClearElement);
	
	//! Setzt den Inhalt eines bestimmten Gridelements.
	inline void set(int intX,int intY,const T& data);
	
	//! Liefert den Inhalt eines bestimmten Gridelements.
	inline T& get(int intX, int intY);

	//! Standarddestruktor.
	virtual ~CGrid();
	
	/*! \brief Zuweisungsoperator.
	 *
	 *	Der berladene Zuweisungsoperator sorgt zunchst dafr, dass
	 *  die Gre des Zielgrids an die Gre des Quellgrids
	 *  angepat wird. Das Kopieren geschieht komponentenweise, um
	 *  eine mliche Untersttzung eines Deepcopy des Elementtyps \b T auszunutzen.
	 * 
	 */
	CGrid<T>& operator=(const CGrid<T>& source)
	{
		
		// Zuweisung an sich selbst abfangen
		if (this == &source) return *this;

		// Gridgre angleichen
		resize(0,0);
		resize(source.Width,source.Height);
		
		// Griddaten Elementweise kopieren, dabei wird davon
		// ausgegangen dass die Elemente des Quellgrids ein 
		// Deep-Copy untersttzen
		for (int i = 0 ; i<Width*Height; i++)
		{

			((T*)Elements)[i] = ((T*) source.Elements)[i];

		}

		// Referenz auf sich selbst zurckgeben
		return *this;

	};

};

/*! 
 *  Der Standardkonstruktor erzeugt ein NULL-Grid. Dieses Grid
 *  hat die Ausmae 0 x 0 und enthlt somit keine Elemente.
 *  Das Elementfreigabeflag wird standardmig auf \b false
 *  gesetzt.
 *
 */
template <class T> CGrid<T>::CGrid() : CAbstractGrid()
{

	// Instanzvariablen initialisieren
	DisposeData = false;		     // Kein freigeben von Elementen
	SizeOfElement = sizeof(T);   // Gre des gespeicherten Elements setzen

};

/*! 
 *  Der zustzliche Konstruktor erzeugt ein Gird mit den 
 *  Ausmaen \b intWidth x \b intHeight. 
 *  Das Elementfreigabeflag wird standardmig auf \b false
 *  gesetzt.
 *  Der Speicherbereich fr die Elementdaten wird durch Nullen
 *  berschrieben.
 *
 */
template <class T> CGrid<T>::CGrid(int intWidth, int intHeight) : CAbstractGrid()
{

	// Instanzvariablen initialisieren
	DisposeData = false;		     // Kein freigeben von Elementen
	SizeOfElement = sizeof(T);   // Gre des gespeicherten Elements setzen
	
	// Gridgre ndern und Inhalt lschen
	resize(intWidth, intHeight);
	clear();

};

/*!
 *  Der Kopierkonstruktor verhlt sich analog zum Zuweisungsoperator.
 *  Zunchst wird die Gre des Zielgrids an die Gre des Quellgrids
 *  angepat. Danach werden die Griddaten Elementweise kopiert, um ein 
 *  mgliches Deepcopy des Elementtyps \b T zu verwenden.
 *
 */
template <class T> CGrid<T>::CGrid(CGrid<T>& objGrid) : CAbstractGrid()
{

	// Instanzvariablen initialisieren
	DisposeData = false;		     // Kein freigeben von Elementen
	SizeOfElement = sizeof(T);   // Gre des gespeicherten Elements setzen

	// Gridgre angleichen
	resize(objGrid.Width,objGrid.Height);
	
	// Griddaten Elementweise kopieren, dabei wird davon
	// ausgegangen dass die Elemente des Quellgrids ein 
	// Deep-Copy untersttzen
	for (int i = 0 ; i<Width*Height; i++)
	{
		((T*)Elements)[i] = ((T*)objGrid.Elements)[i];
	}

};

/*!
 *	Diese Methode wird im Zusammenhang mit \b DisposeData verwendet.
 *  Innerhalb der \b CGrid - Klasse ist sie leer, und hat somit keinerlei Auswirkung.
 *  In abgeleiteten Klassen sollte sie berschrieben werden, um den Speicher freizugeben,
 *  der von einem dynamisch alloziierten Element, oder einer Objektreferenz belegt wird.
 *
 */

template<class T> void CGrid<T>::disposeElement(int intX,int intY)
{
};

/*! 
 * Die Methode ndert die Ausmae des Grids. Dabei bleiben die
 * Elementdaten weitestgehend erhalten. Ebenfalls wird der
 * Speicherbereich fr die Elementdaten durch die Methode
 * reorganisiert, insbesondere bei einer nderung der Grid-Breite.
 * Eventuell neu hinzugekommene Elemente werden mit Null-Werten
 * initialisiert.
 *
 * \remarks
 * Ist das \b DisposeData - Flag auf \b true gesetzt, so 
 * ruft die Methode fr jedes Element, welches aufgrund der
 * Grennderung gelscht werden soll, die Methode \b disposeElement()
 * auf.
 *
 */
template <class T> void CGrid<T>::resize(int intNewWidth, int intNewHeight)
{
	
	// Grenangaben verifizieren
	if (intNewWidth<0) intNewWidth = 0;
	if (intNewHeight<0) intNewHeight = 0;
	if (intNewWidth * intNewHeight == 0 ) intNewWidth = intNewHeight = 0;

	// Prfen, ob sich an den Ausmaen tatschlich etwas gendert hat.
	if ((intNewWidth!=Width) || (intNewHeight!=Height))
	{
	
		// Anzahl der zu kopierenden Spalten und Zeilen berechnen
		int intLinesToCopy = (intNewHeight<Height) ? intNewHeight : Height;
		int intRowsToCopy = (intNewWidth<Width) ? intNewWidth : Width;
		
		// berschssige Elemente freigeben, falls ntig 
		if (DisposeData)
		{

			if (Width>intNewWidth)
			{

				for (int y=0 ; y<intLinesToCopy; y++)
				{
				
					for (int x=intNewWidth ; x<Width ; x++)
					{

						// Element freigeben 
						disposeElement(x,y);

					}

				}

			}

			if (Height>intNewHeight)
			{
				
				for (int y=intNewHeight; y<Height; y++)
				{

					for (int x=0 ; x<Width ; x++)
					{

						disposeElement(x,y);

					}

				}

			}

		}

		// Daten reorganisieren
		if (!isNull() && (intNewWidth*intNewHeight!=0))
		{

			T* ptrLine;
			T* ptrDest;

			// Wird Gridbreite kleiner
			if (intNewWidth<=Width)
			{

				ptrLine = (T*) Elements;
				ptrDest = (T*) Elements;

				// Gridzeilen umkopieren
				for (int y = 0 ; y<intLinesToCopy ; y++)
				{
					
					for (int x = 0 ; x<intRowsToCopy ; x++)
					{

						ptrDest[x]=ptrLine[x];

					}

					// Zur nchsten Gridzeile
					ptrLine+=Width;
					ptrDest+=intNewWidth;

				}

				// Speicher realloziieren
				Elements = realloc(Elements,intNewWidth*intNewHeight*sizeof(T));

			}
			else
			{

					// Speicher realloziieren
					Elements = realloc(Elements,intNewWidth*intNewHeight*sizeof(T));

					ptrLine = ((T*) Elements) + Width*(intLinesToCopy-1)+intRowsToCopy-1;
					ptrDest = ((T*) Elements) + intNewWidth*(intLinesToCopy-1)+intRowsToCopy-1;
		
					// Gridzeilen umkopieren
					for (int y = 0 ; y<intLinesToCopy ; y++)
					{
					
						for (int x = 0 ; x<intRowsToCopy ; x++)
						{

							ptrDest[-x]=ptrLine[-x];

						}

						// Zur nchsten Gridzeile
						ptrLine-=Width;
						ptrDest-=intNewWidth;

					}

			}

		}
		else
		{

			// Neuen Speicher fr Elementdaten alloziieren
			Elements = realloc(Elements,intNewWidth*intNewHeight*sizeof(T));

		}

		// Neu hinzugekommene Elemente zurcksetzen
		if (intNewWidth>Width)
		{
						
			T* ptrLine = (T*) Elements;

			for (int y = 0 ; y<intNewHeight ; y++)
			{

				for (int x = Width ; x<intNewWidth ; x++)
				{

					// Element mit 0 fllen
			  	memset(ptrLine+x,0,sizeof(T));

				}
				ptrLine+=intNewWidth;

			}

		}

		if (intNewHeight >Height)
		{
			
			T* ptrLine = ((T*) Elements) + Height*intNewWidth;

			for (int y = Height ; y<intNewHeight ; y++)
			{

				for (int x = 0 ; x<intNewWidth ; x++)
				{

					// Element mit 0 fllen
					memset(ptrLine+x,0,sizeof(T));

				}
				ptrLine+=intNewWidth;

			};

		};

		// Neue Gre speichern
    Width = intNewWidth;
		Height = intNewHeight;
		
	};

};

/*!
 *  Die Methode berschreibt den Speicherberich fr die 
 *  Elementdaten mit Null-Werten.
 * 
 *  \remarks
 *  Ist das \b DisposeData - Flag gesetzt, so wird zunchst
 *  fr jedes Gridelement die Methode \b disposeElement()
 *  aufgerufen.
 *
 */
template <class T> void CGrid<T>::clear()
{

	if (!isNull())
	{
		// Prfen, ob Elemente freigegeben werden sollen
		if (DisposeData)
		{
			
			for (int y = 0; y<Height ; y++)
			{

				for (int x = 0 ; x<Width ; x++)
				{

					// Element freigeben
					disposeElement(x,y);

				}

			}

		};

		// Speicher auf 0 setzen
		memset(Elements, 0, Width*Height*sizeof(T));

	}

};

/*!
 *  Die Methode lscht das Grid, indem jedes Element mit dem
 *  Inhalt von \b objClearElement berschrieben wird. 
 *
 *  \remarks
 *  Ist das \b DisposeData - Flag gesetzt, so wird zunchst
 *  fr jedes Gridelement die Methode \b disposeElement()
 *  aufgerufen.
 *
 */
template <class T> void CGrid<T>::clear(const T& objClearElement)
{

	if (!isNull())
	{

		// Prfen, ob Elemente freigegeben werden sollen
		if (DisposeData)
		{
			
			for (int y = 0; y<Height ; y++)
			{
				
				for (int x = 0 ; x<Width ; x++)
				{

					// Element freigeben
					disposeElement(x,y);

				}

			}

		};

		// Grid mit objClearElement fllen
		for (int i = 0 ; i<Height*Width ; i++)
		{

			((T*)Elements)[i] = objClearElement;

		}

	}

};

/*!
 *  Weist dem Element mit den Koordinaten \b intX und \b intY \b data als Wert zu.
 *
 *  \note 
 *  Fr einen schnelleren Zugriff findet keine Bereichsberprfung statt.
 */

template <class T> inline void CGrid<T>::set(int intX,int intY,const T& data)
{

	((T*) Elements)[intY*Width+intX] = data;

};

/*!
 *  Gibt den Inhalt des Elements mit den Koordinaten \b intX und \b intY zurck.
 *
 *  \note 
 *  Fr einen schnelleren Zugriff findet keine Bereichsberprfung statt.
 */
template <class T> inline T& CGrid<T>::get(int intX, int intY)
{

	return ((T*) Elements)[intY*Width+intX];

};

/*!
 *
 * Der Standarddestruktor lscht zunchst den Elementspeicherbereich
 * mittels \b clear(). Dadurch wird der Speicher, den ein evtl. dynamisch 
 * alloziiertes Element belegt freigegeben, falls das \b DisposeData - Flag zuvor gesetzt wurde.
 * Danach werden die Gittermae auf 0 x 0 gesetzt, was zur Folge hat, das der
 * Elementspeicherbereich freigegeben wird.
 *
 */
template <class T> CGrid<T>::~CGrid()
{

	// Inhalt lschen
	clear();

	// Speicher freigeben, durch reallokation mit Gre 0
	resize(0,0);

};

#endif

