/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CFrameList.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CFrameList.
 *   
 *  @author Christoph Brzozowski.
 *  
 */

#ifndef cls_CFrameListH
#define cls_CFrameListH

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

#include <stdlib.h>

#include "CKeyFrame.h"
#include "CDynamicArray.h"

// ****************************************************************
//
// Klasse: CFrameList
//
// Zweck : Realisiert eine doppelt verkettete bezglich des
//         Zeitindex sortierte Liste von Keyframes.
//       
// Autor: Christoph Brzozowski
//
// ****************************************************************


/*!
 * @class CFrameList
 *
 * @ingroup bsplineWarpClasses
 *
 * \brief Containerklasse zur Verwaltung von Keyframes beliebigen Typs.
 *
 *  Die Klasse \b CFrameList stellt einen Container dar, der beliebige
 *  Keyframes aufnehmen kann, die von der Klasse \b CKeyFrame abgeleitet worden
 *  sind.
 *  Die Keyframes werden innerhalb des Containers aufsteigend bezglich ihres
 *  Zeitindex sortiert.
 *  Mittels eines Cursors kann die Frameliste in beliebiger Richtung
 *  durchlaufen werden.
 *
 *  \remarks
 *  Die der \b CFrameList -Klasse zugrundeliegende Datenstruktur ist eine
 *  doppeltverkettete sortierte Liste.
 *
 *  \note
 *  Obwohl innerhalb des Containers beliebige Keyframe-Typen gemischt werden
 *  knnen, ist dies nicht empfehlenswert.
 *
 */
class CFrameList
{

protected:
	
	//! Hilfsstruktur zur Verkettung der einzelnen Keyframes.
	struct SFrameListElement
	{

			//! Speichert einen Zeiger auf das Keyframe.
			CKeyFrame* Frame;
			
			//! Speichert einen Zeiger auf das vorherige Listenelement.
			SFrameListElement* Previous;
			
			//! Speichert einen Zeiger auf das nchste Listenelement.
			SFrameListElement* Next;

	};

	//! Speichert einen Zeiger auf das erste Element innerhalb der Liste.
	SFrameListElement* First; 
	
	//! Speichert einen Zeiger auf das letzte Element der Liste.
	SFrameListElement* Last;  
	
	/*! \brief Speichert die Anzahl der Listenelemente, d.h. die Anzahl der Keyframes.
	 *
	 *  \note
	 *  Alle Methoden, die die Anzahl der Elemente innerhalb der Liste ndern
	 *  mssen \b Count entsprechend anpassen.
	 *
	 */
	int Count;								

	//! Framecursor.
	SFrameListElement* Current; 

	//! Hilfsfunktion zum ausklinken von Elementen aus der verketteten Liste.
	void LinkOut(SFrameListElement* Element);

public:
	
	//! Standardkonstruktor.
	CFrameList();

	//! Gibt die Anzahl der Frames zurck.
	int count()
	{

		return Count;

	};

	//! Lsche alle Keyframes aus der Liste.
	void clear();

	//! Fgt ein neues Keyframe in die Liste ein.
	void insert(CKeyFrame* NewFrame);

	//! Entfernt das Keyframe, an dem sich der Framecursor gerade befindet.
	bool remove();

	//! Entfernt das Keyframe mit dem ber \b Timeindex angegebenen Zeitindex.
	bool remove(float TimeIndex);

	//! Gibt das Keyframe zurck, an dem sich der Framecursor gerade befindet.
	CKeyFrame* current()
	{
		if (Current!=NULL) 
		return Current->Frame;
		else
		return NULL;
		
	};

	//! Bewegt den Framecursor zu dem Keyframe mit dem ber \b TimeIndex angegebenen Zeitindex.
	CKeyFrame* current(float TimeIndex);

	//! Bewegt den Cursor zum ersten Frame der Liste.
	CKeyFrame* first()
	{

		Current = First;
		return current();

	};

	//! Bewegt den Cursor zum letzten Frame der Liste.
	CKeyFrame* last()
	{
		Current = Last;
		return current();
	};

	//! Bewegt den Cursor zum nchsten Frame innerhalb der Liste.
	CKeyFrame* next()
	{ 

		if (Current!=NULL) Current = Current->Next;
		return current();

	};

	//! Bewegt den Cursor zum vorherigen Frame innerhalb der Liste.
	CKeyFrame* previous()
	{

		if (Current!=NULL) Current = Current->Previous;
		return current();

	};

	/*! \brief Verschiebt das Frame, an dem sich gerade der Framecursor befindet, 
	 *  um den Wert \b Delta innerhalb der Zeitskala.
	 */
	void shift(float Delta);

	//! Standarddestruktor.
	~CFrameList();

};

#endif
