/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CFrameList.cpp
 *
 *  @brief
 *	Implementation der Klasse \b CFrameList.
 *   
 *  @author Christoph Brzozowski
 *  
 */

#include <stdlib.h>
#include <math.h>

#include "CFrameList.h"

/*! 
 *  Der Standardkonstruktor initialisiert alle Elementzeiger mit NULL, und 
 *  setzt die Anzahl der Elemente auf 0.
 */
CFrameList::CFrameList()
{
	
	// Frameanzahl zurcksetzen
	Count = 0;

	// Elementzeiger zurcksetzen
	First = NULL;
	Last  = NULL;
	Current = NULL;

};

/*!
 *  Die Methode \b insert() fgt das ber \b NewFrame spezifizierte
 *  Keyframe zu der Frameliste hinzu. Dabei wird das Keyframe
 *  an der richtigen Stelle innerhalb der Liste bezglich des
 *  Zeitindex eingefgt, so dass die Liste immer sortiert bleibt.
 *
 *  \note
 *  Existiert bereits ein Frame mit idetischem Zeitindex, wie das
 *  einzufgende Frame, so wird es durch das neue Frame berschrieben.
 *
 */

void CFrameList::insert(CKeyFrame* NewFrame)
{
	
	if (NewFrame!=NULL)
	{

		// Elementzhler erhhen
		Count++;

		// Prfen, ob Liste bereits Elemente enthlt
		if (First!=NULL)
		{
			
			SFrameListElement*  tmpPtr=First;
			
			// Liste nach geeigneter Einfgestelle durchsuchen
			while ((tmpPtr!=NULL) && (tmpPtr->Frame->timeIndex()<NewFrame->timeIndex()))
			{

				tmpPtr=tmpPtr->Next;

			};
			
			// Komplette Liste durchlaufen???
			if (tmpPtr==NULL)
			{
				
				// Element am Ende der Liste anhngen
				SFrameListElement* tmpElement = new SFrameListElement;
				tmpElement->Previous = Last;
				tmpElement->Frame = NewFrame;
				tmpElement->Next = NULL;
				Last->Next = tmpElement;
				Last = tmpElement;

			}
			else // Liste gar nicht durchlaufen???
			if (tmpPtr==First)
			{
				
				// Element am Anfang der Liste anhngen
				SFrameListElement* tmpElement = new SFrameListElement;
				tmpElement->Frame = NewFrame;
				tmpElement->Next = First;
				First->Previous=tmpElement;
				tmpElement->Previous = NULL;
				First = tmpElement;

			}
			else
			{
				
				// Prfen, ob der Zeitindex der Einfgestelle mit dem Zeitindex
				// des einzufgenden Frames bereinstimmt
				SFrameListElement* tmpElement;

				if (fabsf(tmpPtr->Frame->timeIndex()-NewFrame->timeIndex()) < 1E-8)
				{
					
					tmpElement = tmpPtr;
					// Alte Framedaten lschen
					delete tmpPtr->Frame;
					// Frame zuweisen
					tmpPtr->Frame = NewFrame;

				}
				else
				{
					
					// Element in die Liste einhngen
					tmpElement = new SFrameListElement;
					tmpElement->Frame = NewFrame;
					tmpElement->Next = tmpPtr;
					tmpElement->Previous = tmpPtr->Previous;
					tmpPtr->Previous->Next = tmpElement;
					tmpPtr->Previous = tmpElement;

				};

			};

		}
		else
		{

			// Element als erstes und letztes Element der Liste setzen
			First = new SFrameListElement;
			First->Frame = NewFrame;
			First->Next = NULL;
			First->Previous =NULL;
			Last = First;

		};

	};

};

/*! 
 *  Die Methode \b current(float TimeIndex) durchsucht
 *  die Liste nach dem mittels \b TimeIndex
 *  spezifizierten Frame. Wird es gefunden, so 
 *  wird die Framecursorposition auf dieses
 *  Frame gesetzt und ein Zeiger auf das Keyframe 
 *  zurckgegeben. Ansonsten wird NULL zurckgegeben,
 *  und die Position des Framecursors ist undefiniert.
 */
CKeyFrame* CFrameList::current(float TimeIndex)
{

	// Liste von Anfang an durchlaufen und Frame suchen
	Current = First;

	while ((Current!=NULL) && (fabsf(Current->Frame->timeIndex()-TimeIndex) > 1E-8))
	{

		Current = Current->Next;

	};

	// Frame zurckgeben
	if (Current!=NULL)
	{

		return Current->Frame;

	}
	else
	{

		return NULL;

	};

};

/*! 
 *  Die Methode \b LinkOut() hngt ein Element aus der 
 *  doppelt verketteten Liste aus, und passt dabei die
 *  Zeiger \b First, \b Last und \b Current entsprechend
 *  an.
 *
 *  \remarks
 *  Wird gerade das Frame ausgeklinkt an dem sich der
 *  Framecursor befindet, so wird dieser je nach
 *  Lage entweder auf das nchste oder vorherige Frame
 *  gesetzt.
 */
void CFrameList::LinkOut(SFrameListElement* Element)
{

		// Element aus der Liste aushngen
		if (Element->Previous!=NULL) Element->Previous->Next = Element->Next;
		if (Element->Next!=NULL) Element->Next->Previous = Element->Previous;

		// Last- und First-Zeiger anpassen falls ntig
		if (Element->Next==NULL)
		{

			 Last = Element->Previous;

		};
		
		if (Element->Previous==NULL)
		{

			First = Element->Next;

		};
		
		if (Element == Current)
		{

			// Current-Zeiger je nachdem auf vorheriges
			// oder nchstes Element setzen
			if (Element->Previous!=NULL)
			{

				Current = Element->Previous;

			}
			else
			{
				
				if (Element->Next!=NULL)
				{

					Current = Element->Next;

				}
				else
				{
					
					Current = NULL;
				}

			}

		}

};

/*! 
 *  Die Methode \b remove() gibt je nach Erfolg \b true oder \b false zurck.
 *
 *  \remarks
 *  Der von dem Keyframe belegte Speicherplatz wird freigegeben.
 * 
 *  \remarks
 *  Der Framecursor wird nach dem Entfernen des Frames je nach
 *  Lage entweder auf das nchste oder vorherige Frame
 *  gesetzt.
 */
bool CFrameList::remove()
{
	
	if (Current!=NULL)
	{

		// Current-Zeiger zwischenspeichern
		SFrameListElement* tmpElement = Current;

		// Element aus der Liste aushngen
		LinkOut(Current);

		// Framedaten lschen
		delete tmpElement->Frame;

		// Element lschen
		delete tmpElement;

		// Elementzhler erniedrigen
		Count--;

		return true;

	}
	else 
	{
		
		return false;

	};

};

/*! 
 *  Die Methode \b remove(float TimeIndex) gibt je nach Erfolg \b true oder \b false zurck.
 *
 *  \remarks
 *  Der von dem Keyframe belegte Speicherplatz wird freigegeben.
 * 
 *  \remarks
 *  Wird gerade das Frame entfernt an dem sich der
 *  Framecursor befindet, so wird dieser je nach
 *  Lage entweder auf das nchste oder vorherige Frame
 *  gesetzt.
 */
bool CFrameList::remove(float TimeIndex)
{

	// Prfen, ob Elemente in der Liste vorhanden sind
	if (Count>0)
	{

		// Liste nach geeignetem Element durchsuchen
		SFrameListElement* tmpPtr = First;

		while (tmpPtr!=NULL)
		{

			// Stimmen die Zeitindizes berein?
			if( fabsf(tmpPtr->Frame->timeIndex()-TimeIndex) < 1E-8 )
			{

				// Element aus der Liste aushngen
				LinkOut(tmpPtr);
				
				// Framedaten lschen
				delete tmpPtr->Frame;

				// Element lschen
				delete tmpPtr;
				
				// Elementzhler erniedrigen				
				Count--;

				return true;

			};

			tmpPtr = tmpPtr->Next;

		};

	};

	// Aktion nicht erfolgreich
	return false;

};

void CFrameList::shift(float Delta)
{
	
	// Prfen, ob Current-Zeiger gltig
	if ((Current!=NULL) && (fabsf(Delta) > 1E-8))
	{

		// Current-Zeiger zwischenspeichern
		SFrameListElement* tmpCurrent = Current;
		
		// Current-Element aus der Liste aushngen
		LinkOut(Current);

		// Elementzhler erniedrigen, da er nachher beim Einfgen
		// wieder erhht wird
		Count--;

		// Zeitindex verschieben
		tmpCurrent->Frame->TimeIndex += Delta;

		// Element neu in die Liste einhngen und Current-Zeiger auf
		// eingehngtes Element setzen
		insert(tmpCurrent->Frame);
		current(tmpCurrent->Frame->TimeIndex);

		// Temporres Listen-Element freigeben
		delete tmpCurrent;

	}

};

/*! Die Methode \b clear() entfernt alle Keyframes
 *  aus der Liste. Der durch die Keyframes belegte
 *  Speicherplatz wird dabei freigegeben.
 */
void CFrameList::clear()
{

	// Keyframes entfernen und Speicher freigeben
	if (Count>0)
	{
		
		SFrameListElement* tmpPtr = First;
	
		do
		{
			
			// Frame freigeben
			delete(tmpPtr->Frame); 

			SFrameListElement* tmpNext = tmpPtr->Next; 

			// Listenelement freigeben
			delete tmpPtr; 

			// Zum nchsten Element gehen
			tmpPtr = tmpNext; 

		}
		while (tmpPtr!=NULL);

	};

	// Elementzeiger zurcksetzen
	Count = 0;
	First = NULL;
	Last  = NULL;
	Current = NULL;

};

/*! Der Standardestruktor lscht mittels \b clear() alle
 *  Elemente aus der Liste.
 */
CFrameList::~CFrameList()
{
	clear();
};

