/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CDynamicArray.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CDynamicArray.
 *   
 *  @author Christoph Brzozowski.
 *  
 */

#ifndef cls_CDynamicArrayH
#define cls_CDynamicArrayH

#if     _MSC_VER > 1000
#pragma once
#endif

#include <stdlib.h>
#include <memory.h>
#include <stdexcept>
using std::out_of_range;

/*!
 *
 *  @class CDynamicArray
 *
 *  @ingroup bsplineWarpClasses
 * 
 * \brief Dynamisches eindimensionales Array.
 *
 *  Das Klassentemplate \b CDynamicArray implementiert ein dynamisches Array,
 *  welches Elemente vom Datentyp \b T speichert.  Da der Speicher dynamisch
 *  alloziiert wird, kann die gre des Arrays zur Laufzeit gendert werden. 
 * 
 *  \note
 *  Die Klasse eignet sich nicht zur Aufbewahrung von Referenzen oder dynamisch
 *  alloziierten Daten, da bei einer Grennderung des Arrays, insbesondere
 *  einer Verkleinerung, der Speicher fr die berschssigen Elemente nicht
 *  freigegeben wird.
 */
template <class T> class CDynamicArray
{
	
protected:
	
	//! Speichert die Anzahl der Elemente im Array.
	int Size;	  
	//! Speichert die Elemente des Arrays.
	T* Elements; 

public:

	//! Standardkonstruktor.
	CDynamicArray();

	//! Zusatzkonstruktor.
	CDynamicArray(int size);

	//! Kopierkonstruktor.
	CDynamicArray(const CDynamicArray<T>& array);

	//! Gibt zurck, ob das Array ein NULL-Array ist.
	inline bool isNull();

	//! Liefert die Anzahl der Elemente im Array zurck.
	inline int getSize();

	//! ndert die Gre des Arrays.
	void resize(int newSize);

	//! Liefert einen Zeiger auf den Speicherbereich zurck, welcher die Elemente des Arrays beinhaltet.
	inline T* get();

	//! Liefert das Element mit dem Index \b index zurck.
	inline T& get(int index);

	//! Speichert \b data an der Stelle \b index im Array.
	inline void set(int index,const T& data);
  
	/*! \brief "[]"-Operator (Indizierter Zugriff).
	 *
	 *  Der Operator gibt eine Referenz auf das Element mit dem Index \b index zurck. 
	 *  Der indizierte Zugriff kann also sowohl schreibend, als auch lesend verwendet werden.
	 *
	 *  \note
	 *  Fr einen schnelleren Zugriff wurde auf eine Bereichsberprfung verzichtet!
	 *
	 */
	T& operator[](int index)
	{	
		return Elements[index];
	};
	
	//! Standarddestruktor.
	~CDynamicArray();

};

/*!
 *  Der Standardkonstruktor erzeugt ein leeres Array mit Null Elementen.
 */
template <class T> CDynamicArray<T>::CDynamicArray()
{
	
	// Member-Variablen initialisieren	
	Size = 0;
	Elements = NULL;

};

/*!
 *  Der Zusatzkonstruktor erzeugt ein Array mit \b size Elementen.
 *  Der Speicherbereich fr die Arraydaten wird dabei mit Nullen
 *  berschrieben.
 */
template <class T> CDynamicArray<T>::CDynamicArray(int size)
{
	
	// Member-Variablen initialisieren	
	Size = 0;
	Elements = NULL;

	// Speicher durch nderung der Gre alloziieren
	resize(size);	

};

/*!
 *  Der Kopierkonstruktor erzeugt ein Array auf der Basis eines anderen
 *  \b CDynamicArray - Objekts.
 *
 *  \remarks
 *  Die Daten werden nicht Elementweise kopiert. Stattdessen wird einfach
 *  der Inhalt des Speicherbereichs des Quellarrays mittels \b memmove() 
 *  in den Speicherbereich des Zielarrays kopiert. Falls die Arrayelemente 
 *  Referenzen beinhalten, oder selbst Referenzen sind, so zeigen diese anschlieend
 *  Sowohl im Quellarray, als auch im Zielarray auf dieselben Daten.
 */
template <class T> CDynamicArray<T>::CDynamicArray(const CDynamicArray<T>& array)
{

	// Member-Variablen initialisieren
	Size = 0;
	Elements = NULL;

	// Speicher durch nderung der Gre alloziieren, und zwar soviel,
	// damit die Elemente des Quell-Arrays darin Platz finden
	resize(array.getSize());

	// Elementdaten kopieren
	memmove(Elements,array.get(),Size*sizeof(T));

};

/*!
 * Die Funktion gibt <true> zurck, falls das Array eine Gre
 * von Null Elementen aufweist, sonst <false>.
 */
template <class T> inline bool CDynamicArray<T>::isNull()
{
	return ((Size==0) && (Elements==NULL));
};

template <class T> inline int CDynamicArray<T>::getSize()
{
	return Size;
};

/*!
 * Die Funktion ndert die Gre des Arrays. Beim Vergrern
 * des Arrays bleibt sein Inhalt vollstndig erhalten, der Inhalt
 * der neuen Elemente ist undefiniert. Beim Verkleinern des
 * Arrays bleiben nur die Elemente mit einem Index <\b newSize
 * erhalten.
 *
 * \note
 * Sollte es sich bei den Elementen um Referenzen handeln, so
 * wird der Speicher auf den sie zeigen von der Routine nicht
 * freigegeben!
 */
template <class T> void CDynamicArray<T>::resize(int newSize)
{

	// Prfen, ob spezifizierte Gre sich von der aktuellen
	// unterscheidet
	if (newSize!=Size)
	{

		// Neue Gre gltig ?
		if (newSize>0)
		{

			// Gre setzen und Arrayspeicher realloziieren
			Size = newSize;
			Elements = (T*)realloc(Elements,Size*sizeof(T));

		}
		else
		{

			// NULL-Array erzeugen bei inkorrekter Angabe
			// der Gre, Element-Speicher freigeben
			Size = 0;
			Elements = (T*) realloc(Elements,0);

		};

	};

};

template <class T> inline  T* CDynamicArray<T>::get()
{
	return Elements; 
};

/*! \note Es findet eine Bereichsberprfung statt! Liegt der Index ausserhalb des gltigen Bereichts, so wird
 *  eine Exception des Typs \b out_of_range() ausgelst.
 */
template <class T> inline T& CDynamicArray<T>::get(int index)
{
	
	// Bereich prfen

	if ((index>=0) && (index<Size))
	{
		// Array-Element zurckgeben
		return Elements[index];
	}
	else
	{
		throw out_of_range("Der Array-Index liegt auerhalb des zulssigen Bereichs!");
	};

};

/*! \note Es findet eine Bereichsberprfung statt! Liegt der Index ausserhalb des gltigen Bereichts, so wird
 *  eine Exception des Typs \b out_of_range() ausgelst.
 */
template <class T> inline void CDynamicArray<T>::set(int index, const T& data)
{

	// Bereich prfen
	if ((index>=0) && (index<Size))
	{
		// Array-Element setzen
		Elements[index] = data;
	}
	else
	throw out_of_range("Der angegebene Array-Index liegt auerhalb des zulssigen Bereichs");

};

/*!
 *  Der Destruktor gibt den fr die Arrayelemente reservierten Speicher wieder frei.
 */
template <class T> CDynamicArray<T>::~CDynamicArray()
{

	// Arrayspeicher wieder freigeben, durch realloziieren
	// mit der Gre 0
	resize(0);

};

#endif

