/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CControlGridDeformer.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CControlGridDeformer.
 *   
 *  @author Christoph Brzozowski.
 *  
 */

#ifndef cls_CControlGridDeformerH
#define cls_CControlGridDeformerH

#if     _MSC_VER > 1000
#pragma once
#endif

#include <stdlib.h>
#include <stdexcept>
using std::invalid_argument;

#include "CGridDeformer.h"
#include "CVector.h"

/*!
 *
 *  @class CControlGridDeformer
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Abstrakte Basisklasse fr Deformationsalgorithmen die auf \b CGrid-
 * Objekten operieren und mittels eines Kontrollgitters gesteuert werden.
 *
 *  Die Klasse ist von \b CGridDeformer abgeleitet und erlaubt
 *  natrlich die Spezifikation eines Quell- und eines Zielgitters.
 *  Fr die Templateparameter \b N und \b T gelten dieselben Bedingungen, wie fr
 *  die \b CGridDeformer -Klasse.
 *  Zustzlich kann einer \b CControlGridDeformer -Klasse auch ein
 *  Kontrollgitter zugewiesen werden.
 *
 *  \note
 *  Es ist nicht zwingend erforderlich, dass das Kontrollgitter dieselben
 *  Ausmae aufweist, wie das Quell- und das Zielgitter, sie mssen jedoch
 *  mindestens 2 x 2 betragen.
 *
 *  \note
 *  Die Methode \b deform() bleibt in dieser Klasse weiterhin unimplementiert.
 *
 */
template <int N, class T> class CControlGridDeformer : public CGridDeformer<N,T>
{

protected:

	//! Speichert einen Zeiger auf das Kontrollgitter.
	CGrid< CVector<N,T> >* ControlGrid;    

public:

	//! Standardkonstruktor.
	CControlGridDeformer();

	//! Setzt den Zeiger auf das Kontrollgitterobjekt.
	virtual void setControlGrid(CGrid< CVector<N,T> >* objControlGrid);

	//! Gibt eine Referenz auf das Kontrollgitterobjekt zurck.
	inline CGrid< CVector<N,T> >& controlGrid();

};

/*!
 * Der Standardkonstruktor setzt alle Gitterzeiger auf NULL.
 *
 */
template <int N, class T> CControlGridDeformer<N,T>::CControlGridDeformer():CGridDeformer<N,T>()
{

	ControlGrid = NULL;

};

/*!
 * Die Methode \b setControlGrid() setzt einen Zeiger auf das 
 * zu verwendende Kontrollgitter. 
 *
 * \note 
 * Die Ausmae des Kontrollgitters drfen sich von den Ausmaen
 * des Quell- und des Zielgitters unterscheiden, mssen jedoch
 * mindestens 2 x 2 betragen, da sonst eine Exception vom Typ
 * \b invalid_argument ausgelst wird.
 *
 */

template <int N, class T> void CControlGridDeformer<N,T>::setControlGrid(CGrid< CVector<N,T> >* objControlGrid)
{

	if (objControlGrid!=NULL)
	{

		// Prfen, ob Kontrolgitter > 2x2 ist
		if ((objControlGrid->width()<2) || (objControlGrid->height()<2))
		{

			// Exception auslsen
			throw invalid_argument("Die Gre des Kontrollgitters mu mindestens 2x2 sein!"); 

		}

	}

	// Kontrollgitterreferenz zuweisen
	ControlGrid = objControlGrid;

};

template <int N, class T> inline CGrid< CVector<N,T> >& CControlGridDeformer<N,T>::controlGrid()
{

	return ControlGrid; 

};

#endif
