/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/**
 * @defgroup bsplineWarpClasses B-Spline Warp Effect classes
 *
 */

/*! @file CBSplineWrapperEffect.h
 *
 *  @brief
 *	Deklaratinen der Klasse \b CBSplineWrapperEffect
 *   
 *  @author Christoph Brzozowski
 *     
 */

#ifndef CBSPLINEWRAPPEREFFECT_H
#define CBSPLINEWRAPPEREFFECT_H

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

// Systembibliotheken
#include <stdio.h>
#include <stdlib.h>

// Vom Effekt verwendete Klassenheader
#include "CGrid.h"
#include "CVector.h"
#include "CGridPainter.h"
#include "CReferenceList.h"
#include "CControlGridDeformer.h"
#include "CFrameList.h"
#include "CVertexGridKeyFrame.h"
#include "CGridInterpolator.h"

// Lokale Headerdateien
#include "../main/GLEffect.h" 

//---------------------------------------------------------------------------
//  Vorwrtsreferenzen
//---------------------------------------------------------------------------

// QT-Steuerelemente
class QTimer;
class QSlider;
class QLabel;
class QCheckBox;
class QPushButton;
class QUnclosableDialog;
class QKeyFrameBar;
class QSpinBox;
class QPopupMenu;
class QTable;

// QT-Geometriemanagment-Steuerelemente
class QVBox;
class QHBox;
class QVGroupBox;
class QGrid;
class QGridLayout;

/*!
 *
 * @class CBSplineWrapperEffect
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Implementiert den B-Spline-Wrapper-Effekt.
 *
 *  Die Klasse \b CBSplineWrapperEffect implementiert den 
 *  B-Spline-Wrapper-Effekt mit seiner gesamten funktionalitt.
 *  Da sie von \b GLEffect abgeleitet ist, kann sie leicht in 
 *  das GLFramework integriert werden.
 *
 */
class CBSplineWrapperEffect: public GLEffect
{
	Q_OBJECT

//---------------------------------------------------------------------------
//  ffentliche Instanz-Enumerations.
//---------------------------------------------------------------------------
public:

	/*! \enum enmAnimationMode
	 * 
	 *  \brief Animationsmoduskonstanten.
	 *
	 *  Die Enumeration \b enmAnimationMode enthlt alle Konstanten, die den
	 *  Animationsmodus der Klasse \b CBSplineWrapperEffect steuern.
	 */

	/*! \var enuAnimationMode amOnce
	 *  Einfacher Abspielmodus. Die Animation wird nur einmal abgespielt.
	 */
	/*! \var enuAnimationMode amLoop
	 *  Die Animation wird immer wieder von vorne abgespielt.
	 */
	/*! \var enuAnimationMode amPingPong
	 *  Die Animation wird wiederholt, der Animtionscursor luft immer wieder hin und zurck.
	 */
	enum enmAnimationMode {amOnce, amLoop, amPingPong};

	/*! \enum enmTranslationMode
	 * 
	 *  \brief Manipulationsmoduskonstanten.
	 *
	 *  Die Enumeration \b enmTranslationMode enthlt alle Konstanten, die den
	 *  Manipulationsmodus der Klasse \b CBSplineWrapperEffect fr Kontrolvertices steuern.
	 */
	/*! \var enmTranslationMode tmMove
	 *  Vertex-Verschiebemodus.
	 */
	/*! \var enmTranslationMode tmScale
	 *  Vertex-Skalierungsmodus.
	 */
	/*! \var enmTranslationMode tmRotate
	 *  Vertex-Rotationsmodus.
	 */
	enum enmTranslationMode {tmMove, tmScale, tmRotate};

	/*! \enum enmDeformationMode
	 * 
	 *  \brief Deformationsmoduskonstanten.
	 *
	 *  Die Enumeration \b enmDeformationMode enthlt alle Konstanten, die den
	 *  Deformationsmodus der Klasse \b CBSplineWrapperEffect steuern.
	 */
	/*! \var enmDeformationMode dmLinear
	 *  Linearer Deformationsmodus.
	 */
	/*! \var enmDeformationMode dmLinear
	 *  B-Spline-Deformationsmodus.
	 */
	enum enmDeformationMode {dmLinear,dmBSpline};

	/*! \enum enmEffectMode
	 * 
	 *  \brief Effektmoduskonstanten.
	 *
	 *  Die Enumeration \b enmEffectMOde enthlt alle Konstanten, die den
	 *  Effektmodus der Klasse \b CBSplineWrapperEffect steuern.
	 */
	/*! \var enmEffectMode emKeyFrame
	 *  Key-Frame-Modus.
	 */
	/*! \var enmEffectMode emMorphing
	 *  Morphing-Modus.
	 */
	enum enmEffectMode {emKeyFrame,emMorphing};

	/*! \enum enmTranslationState
	 * 
	 *  \brief Translationszustandskonstanten.
	 *
	 *  Die Enumeration \b enmTranslationState enthlt alle Konstanten, die den
	 *  aktuellen Bearbeitungszustand der Klasse \b CBSplineWrapperEffect angeben.
	 */
	/*! \var enmTranslationState tsNothing
	 *  Nichts passiert...
	 */
	/*! \var enmTranslationState tsMoving
	 *  Kontrolvertices werden gerade verschoben.
	 */
	/*! \var enmTranslationState tsScaling
	 *  Kontrolvertices werden gerade skaliert.
	 */
	/*! \var enmTranslationState tsRotating
	 *  Kontrolvertices werden gerade rotiert.
	 */
	/*! \var enmTranslationState tsBoxing
	 *  Ein Selektionsrahmen wird gearde gezogen.
	 */
	/*! \var enmTranslationState tsInterpolating
	 *  Keyframes werden gerade Interpoliert.
	 */
	enum enmTranslationState {tsNothing, tsMoving, tsScaling, tsRotating, tsBoxing, tsInterpolating};

	/*! \enum enmMorphingEditState
	 * 
	 *  \brief Morphingbearbeitungskonstanten.
	 *
	 *  Die Enumeration \b enmMorphingEditState enthlt alle Konstanten, die den
	 *  aktuellen Morphingbearbeitungszustand der Klasse \b CBSplineWrapperEffect angeben.
	 */
	/*! \var enmMorphingEditState esSource
	 *  Es wird gerade das Quellgitter bearbeitet.
	 */
	/*! \var enmMorphingEditState esDestination
	 *  Es wird gerade das Zielgitter bearbeitet.
	 */
	enum enmMorphingEditState {esSource,esDestination};


	/*! \enum enmKnotMatrixType
	 * 
	 *  \brief Knotenmatrixtypkonstanten.
	 *
	 *  Die Enumeration \b enmKnotMatrixType enthlt alle Konstanten, die den
	 *  aktuellen Knotenmatrixbearbeitungszustand der Klasse \b CBSplineWrapperEffect angeben.
	 */
	/*! \var enmKnotMatrixType kmCol
	 *  Es wird gerade die Spaltenmatrix bearbeitet.
	 */
	/*! \var enmKnotMatrixType kmRow
	 *  Es wird gerade die Zeilenmatrix bearbeitet.
	 */
	enum enmKnotMatrixType {kmCol,kmRow};

//---------------------------------------------------------------------------
//  ffentliche Methoden
//---------------------------------------------------------------------------
public:

	//! Standardkonstruktor.
	CBSplineWrapperEffect(GLFramework* parent);
	
	//! Standarddestruktor.
	virtual ~CBSplineWrapperEffect();
	
	//! Erzeugt die vom Effekt zu seiner Parameterisierung bereitgestellten Steuerelemente.
	void createControls(QWidget* parentWindow);

	//! Versteckt den Effekt.
  virtual void hide();

	//! Macht den Effekt sichtbar.
	virtual void show();
		
	//! Startet die Animation.
	virtual void stop();
	
	//! Stoppt die Annimation.
	virtual void play();
	
	//! Setzt den Effekt zurck.
	virtual void reset();

	//! Pausiert die Animation.
	virtual void pause();
	
	//! Rendert den Effekt.
	virtual void render();
	
	//! Event-Handler.
	virtual void mouseMoveEvent(QMouseEvent*);
	//! Event-Handler.
	virtual void mousePressEvent(QMouseEvent*);
	//! Event-Handler.
	virtual void mouseReleaseEvent(QMouseEvent*);
	//! Event-Handler.
	virtual void keyPressEvent(QKeyEvent*);
	
signals:

protected slots:  

	//! Whlt alle Kontrolpunkte an.
	void selectAll();
	//! Whlt alle selektierten Kontrollpunkte ab.
	void unselectAll();
	
	//! Setzt die Kontrolpunktsperre.
	void setBorderLock(bool Enabled);

	//! Setzt den Zeichemodus fr das Bild.
	void setPaintImage(bool Enabled);

	//! Setzt den Zeichenmodus fr das Bildgitter.
	void setPaintImageMesh(bool Enabled);

	//! Setzt den Zeichenmodus fr das Kontrollgitter.
	void setPaintControlMesh(bool Enabled);

	//! Setzt den Zeichenmodus fr die Kontrollgitterpunkte.
	void setPaintControlMeshVertices(bool Enabled);
	
	//! Setzt die vertikale Auflsung des Kontrollgitters.
	void setControlMeshVerticalResolution(int NewResolution);

	//! Setzt die horizontale Auflsung des Kontrollgitters.
	void setControlMeshHorizontalResolution(int NewResolution);

	//! Setzt die horizontale Auflsung des Bildgitters.
	void setImageMeshHorizontalResolution(int NewResolution);

	//! Setzt die vertikale Auflsung des Bildgitters.
	void setImageMeshVerticalResolution(int NewResolution);

	//! Setzt den Animationsmodus.
	void setAnimationMode(int Mode);

	//! Setzt den Vertexmanipulationsmodus.
	void setTranslationMode(int Mode);
	
	//! Setzt den Deformationsalgorithmus.
	void setDeformationMode(int mode);

	//! Setzt den Effektmodus
	void setEffectMode(int Mode);

	//! Setzt die zu bearbeitende Knotenmatrix.
	void setActiveKnotMatrix(int Matrix);

	//! Setzt die Splineordnung.
	void setSplineOrder(int Order);

	//! Setzt die Animationsgeschwindigkeit.
	void setAnimationSpeed(int Speed);

	//! Fgt ein Keyframe ein.
	void insertKeyFrame();

	//! Lscht das markierte Keyframe.
	void deleteKeyFrame();

	//! Springt mit dem Framecursor an die Stelle 0.
	void gotoZero();

	//! Springt mit dem Framecursor zum ersten Keyframe.
	void gotoFirstKeyFrame();

	//! Springt mit dem Framecursor zum nchsten Keyframe.
	void gotoNextKeyFrame();

	//! Springt mit dem Framecursor zum letzten Keyframe.
	void gotoLastKeyFrame();

	//! Springt mit dem Framecursor zum vorherigen Keyframe.
	void gotoPreviousKeyFrame();

	//! Kopiert das Kontrollgitter in das Morphingquellgitter.
	void controlMeshToSource();

	//! Kopiert das Kontrollgitter in das Morphingzielgitter.
	void controlMeshToDestination();

	//! Kopiert das Morphingquellgitter in das Kontrolgitter.
	void sourceToControlMesh();

	//! Kopiert das Morphingzielgitter in das Kontrolgitter.
	void destinationToControlMesh();

	//! Lscht alle gesetzten Keyframes.
	void clearKeyFrames();

	//! Setzt das Kontrollgitter zurck.
	void resetControlMesh();

	//! Setzt die Knotenmatrizen zurck.
	void resetKnotMatrix();

private slots:

	//! Beendet den Interpolationsvorgang.
	void finishInterpolate();
	//! Interpoliert zwischen den Keyframes an der Stelle \b TimeIndex.
	void interpolate(float TimeIndex);
	//! Animiert den Effekt.
	void animate();
	//! Aktualisiert die bearbeitete Knotenmatrix an der Stelle \b row, \b col.
	void updateKnotMatrixValue(int row, int col);

//---------------------------------------------------------------------------
//  Private und Geschtzte Methoden
//---------------------------------------------------------------------------

protected:

	//! Initialisiert die Effektparameter und -zustnde.
	virtual void initialize(); 
	
	//! Versetzt die GUI in einen Initialzustand.
	virtual void initializeGUI();

	//! Berechnet die Koordinaten des Mauszeigers auf der Kontrollgitterebene.
	void getControlPlaneCoordinates();

	//! Berechnet den Index des Kontrollgitterpunkts.
	void getControlPointGridIndex();

	//! Aktiviert oder deaktiviert das Kontrollgitterauflsungsinterface.
	void enableControlMeshResolutionPanel(bool Enabled);

	//! Aktiviert oder deaktiviert das Knotenmatrixbearbeitungsinterface.
	void enableKnotMatrixPanel(bool Enabled);

	//! ndert die Gre des Kontrolgitters.
	void resizeControlMesh(int newWidth, int newHeight);

	//! ndert die Gre des Bildgitters.
	void resizeImageMesh(int newWidth, int newHeight);

	//! Setzt den aktiven Deformationsalgorithmus.
	void setActiveDeformer(CControlGridDeformer<2,GLfloat>* newDeformer);

	//! Whlt alle Knotrollpunkte innerhalb der gezogenen Selektierungsbox an.
	void selectBoxedVertices(bool clear = true);

	//! Schrnkt die Bewegungsmglichkeiten der Randkontrollgitterpunkte ein.
	void lockBorderVertices();

	//! Erzeugt ein regelmiges Gitter.
	void buildRegularMesh(CGrid< CVector<2,GLfloat> >* &Mesh, GLfloat origX, GLfloat origY, GLfloat width, GLfloat height, int ResH, int ResV);

	//! Macht ein bestehendes Gitter regelmig.
	void makeRegularMesh(CGrid< CVector<2,GLfloat> >* &Mesh, GLfloat origX, GLfloat origY, GLfloat width, GLfloat height);

	//! Berechnet den Schnittpunkt zwischen einem Strahl und einer Ebene.
	bool getPlaneRayIntersection(CVector<3,float> rayOrigin,
															 CVector<3,float> rayDirection,
															 CVector<3,float> planeOrigin,
															 CVector<3,float> planeU,
															 CVector<3,float> planeV,
															 float &resultU, 
															 float &resultV);

	//! Liefert die Gitterindices des durch die Maus angewhlten Kontrollpunktes.
	bool getControlPointAtMouse(float threshold, float mx, float my, int &pntCol, int &pntRow);

	//! Selektiert einen bestimmten Kontrollpunkt.
	void selectControlPoint(int x, int y);

	//! Verschiebt die selektierten Kontrollpunkte um einen Delta-Vektor.
	void moveSelectedControlPoints(CVector<2,GLfloat> Delta);
	
	//! Rotiert die selektierten Punkte um das angegebene Rotationszentrum mit dem gegebenen Winkel.
	void rotateSelectedControlPoints(float Angle, CVector<2,GLfloat> Center);

	//! Skaliert die selektierten Punkte um das angegebene Skalierungszentrum mit den gegebenen Skalierungsfaktoren.
	void scaleSelectedControlPoints(float stretchX, float stretchY, CVector<2,GLfloat> Center);

	//! Trennt die Gitterverbindungen zu anderen Objekten.
	void resolveGridBindings();

	//! Verbindet alle Gitter mit dem aktiven Deformationsalgorithmus.
	void bindActiveDeformer();

	//! Trennt die Verbindungen des aktiven Deformationsalgorithmus mit den Gittern.
	void unbindActiveDeformer();

	//! Initialisiert den Keyframe-Modus.
	void initializeKeyFrameMode();

	//! Initialisert den Morphing-Modus.
	void initializeMorphingMode();

	//! Berechnet die Texelkoordinaten eines Gitters neu.
	void recalculateTexelCoordinates(CGrid< CVector<2,GLfloat> >& ImageMesh, CGrid< CVector<2,GLfloat> >& TexelMesh);

	//! Kopiert eine Knotenmatrix in das Tabellensteuerelement.
	void knotMatrixToTable(enmKnotMatrixType matrixType);

	//! Berechnet die Texelkoordinaten des Morphingquell- und Zielgitters neu.
	void recalculateMorphingTexelCoordinates();

private:
	
//---------------------------------------------------------------------------
//  Private und geschtzte Instanzvariablen
//---------------------------------------------------------------------------

protected:

	//! Speichert den Typ der gerade bearbeiteten Knotenmatrix 
	enmKnotMatrixType ActiveKnotMatrix;

	//! Zeitindex fr die Morphingberechnung
	float MorphingTimeIndex;

	//! Speichert den Animationsmodus des Effekts.
	enmAnimationMode AnimationMode;

	//! Speichert die Animationsgeschwindigkeit des Effekts.
	int AnimationSpeed;

	//! Speichert die Abspielrichtung des Effekts.
	float AnimationDirection;

	//! Speichert eine Liste der Kontrollgitterkeyframes.
	CFrameList KeyFrameList;

	//! Speichert die Kurvenordnung.
	int CurveOrder;

	//! Speichert das Handle des OpenGL-Haupttexturobjekts.
	GLuint texture;

	//! Feld mit zur Verfgung stehenden Deformationsalgorithmen.
	CReferenceList< CControlGridDeformer<2,GLfloat> > Deformers;

	//! Speichert eine Referenz auf den gerade aktiven Deformationsalgorithmus.
	CControlGridDeformer<2,GLfloat>* ActiveDeformer;

	//! Speichert den Mainpulationsmodus der Kontrollpunkte.
	enmTranslationMode TranslationMode;
	
	//! Speichert den Deformationsmodus.
	enmDeformationMode DeformationMode;

	//! Speichert den EFfektmodus.
	enmEffectMode EffectMode;

	//! Speichert die Farbe des Bildgitters.
	CVector<4,GLfloat> ImageMeshColor;

	//! Speichert die Farbe des Kontrollgitters.
	CVector<4,GLfloat> ControlMeshColor;

	//! Speichert die Farbe der Kontrollpunkte.
	CVector<4,GLfloat> ControlMeshVertexColor;

	//! Speichert die Farbe der gewhlten Kontrollpunkte.
	CVector<4,GLfloat> SelectedControlMeshVertexColor;

	//! Speichert eine Referenz auf das Kontrollgitter.
	CGrid< CVector<2,GLfloat> >* ControlMesh;

	//! Speichert eine Referenz auf das Kontrollpunktfarbgitter.
	CGrid< CVector<4,GLfloat> >* ControlPointColorGrid;

	//! Speichert eine Referenz auf das Bildgitter.
	CGrid< CVector<2,GLfloat> >* ImageMesh;

	//! Speichert eine Referenz auf das Arbeitstexelgitter.
	CGrid< CVector<2,GLfloat> >* TexelMesh;

	//! Speichert eine Referenz auf das Morphing-Quelltexelgitter.
	CGrid< CVector<2,GLfloat> >* TexelMeshA;
	
	//! Speichert eine Referenz auf das Morphing-Zeiltexelgitter.
	CGrid< CVector<2,GLfloat> >* TexelMeshB;

	//! Objekt zum Zeichnen des Kontrollgitters.
	CGridPainter ControlMeshPainter;

	//! Objekt zum Zeichnen des Bildgitters.
	CGridPainter ImageMeshPainter;

	//! Liste mit selektierten Kontrollpunkten.
	CReferenceList< CVector<2,GLfloat> > SelectedControlPoints;
	
	//! Bestimmt, ob die Bewegung Randkontrollpunkte eingeschrnkt ist.
	bool LockBorder;

	//! Bestimmt, ob das Bild gezeichnet werden soll.
	bool PaintImage;
	
	// Bestimmt, ob das Bildgitter gezeichnet werden soll.
	bool PaintImageMesh;

	// Bestimmt, ob das Kontrollgitter gezeichnet werden soll.
	bool PaintControlMesh;

	// Bestimmt, ob die Kontrollpunkte gezeichnet werden sollen.
	bool PaintControlMeshVertices;

// -------------------------------------------------------
// QT-Widgets
// -------------------------------------------------------

	QPopupMenu* popTranslationMode;
	QPopupMenu* popDeformationMode;
	QPopupMenu* popEffectMode;
	QPopupMenu* popAnimationMode;
	QPopupMenu* popMatrixType;

	// --- Tool window ---------------------------------------

	QUnclosableDialog* dlgToolWindow;				// Additional tool window
	QGridLayout* layToolWindow;	  // It's layout
	QVBox* pnlToolWindow;				  // It's main panel

	// --- Mode panel ----------------------------------------
	
	// Layout and grouping elements

	QHBox* pnlMode;    					  // Deformation & Translation 
																// mode panel
	
	// Buttons

	QPushButton* btnTranslationMode;
	QPushButton* btnDeformationMode;
	QPushButton* btnEffectMode;

	// --- Mesh parameter panel ------------------------------

	// Layout and grouping elements
	
	QHBox* pnlGridParam;					// Mesh parameter panel
	QVGroupBox* grpOrder;					// Curve order group
	QVGroupBox* grpResolution;		// Mesh resolution panel
	QHBox* hbxImageMeshRes;				// Image mesh resolution panel
	QHBox* hbxControlMeshRes;			// Control mesh resolution panel
	QVGroupBox* grpNodeMatrix;		// Knot matrix panel

	// Labels

	QLabel* lblOrder;		
	QLabel* lblControlMesh;
	QLabel* lblImageMesh;
	QLabel* lblHorResControlMesh;	
	QLabel* lblVerResControlMesh;
	QLabel* lblHorResImageMesh;
	QLabel* lblVerResImageMesh;

	// Spin boxes

	QSpinBox* spnOrder;
	QSpinBox* spnHResCMesh;
	QSpinBox* spnVResCMesh;
	QSpinBox* spnHResIMesh;
	QSpinBox* spnVResIMesh;

	// Check boxes
	QCheckBox* cbxLockBorder;

	// Buttons
	QPushButton* btnResetMesh;

	// --- Knot matrix panel ---------------------------------

	// Layout and grouping elements
	QHBox* pnlKnotMatrixNavigation;

	// Buttons
	QPushButton* btnResetKnotMatrix;
	QPushButton* btnMatrixType;
	
	// Table editor
	QTable* tabKnotMatrix;

	// --- Effect timer --------------------------------------

	QTimer* tmrEffectTimer;

	// --- Key frame panel -----------------------------------

	// Layout and grouping elements

	QHBox* pnlFrameNavigation;
	QHBox* pnlFrameBar;
	QGrid* pnlFrameGrid;
	QHBox* pnlMorphingNavigation;

	// User defined keyframe bar widget

	QKeyFrameBar* usrKeyFrameBar;

	// Buttons

	QPushButton* btnFirst;
	QPushButton* btnPrevious;
	QPushButton* btnNext;
	QPushButton* btnLast;
	QPushButton* btnDelete;
	QPushButton* btnInsert;
	QPushButton* btnZero;
	QPushButton* btnClear;

	QPushButton* btnMorphToSource;
	QPushButton* btnMorphFromSource;
	QPushButton* btnMorphToDest;
	QPushButton* btnMorphFromDest;

	QPushButton* btnZoomIn;
	QPushButton* btnZoomOut;
	QPushButton* btnScrollLeft;
	QPushButton* btnScrollRight;

	// --- Paint mode widgets --------------------------------

	// Groupbox

	QVGroupBox* grpPaintMode;
	
	// Checkboxes

	QCheckBox* cbxPaintImage;
	QCheckBox* cbxPaintImageMesh;
	QCheckBox* cbxPaintControlMesh;
	QCheckBox* cbxPaintControlMeshVertices;

	// --- Animation options widgets -------------------------

	QVGroupBox* grpAnimationOptions;
	QPushButton* btnAnimationMode;
	QSlider* sldAnimationSpeed;
	QHBox* hbxFrameLoop;
	QLabel* lblFrameLoopCount;
	QSpinBox* spnFrameLoop;

private:

	//! Speichert den Morphingbearbeitungszustand.
	enmMorphingEditState MorphingEditState;

	//! Speichert den aktuellen Kontrollpunktbearbeitungszustand.
	enmTranslationState TranslationState;

	//! Horizontale Mauskoordinate auf der Kontrollgitterebene.
	float ControlGridPlaneU;
	//! Vertikale Mauskoordinate auf der Kontrollgitterebene.
	float ControlGridPlaneV;

	//! Horizontale Backupebenenoordinate. Wird fr Berechnung von Verschiebungsdeltawerten bentigt.
	float OldControlGridPlaneU;
	//! Vertikale Backupebenenoordinate. Wird fr Berechnung von Verschiebungsdeltawerten bentigt.
	float OldControlGridPlaneV;

	//! Horizontale Ebenenzentrumskoordinate. Wird beim Rotieren und Skalieren von Kontrollgitterpunkten verwendet.
	float TranslationCenterU;
	//! Vertikale Ebenenzentrumskoordinate. Wird beim Rotieren und Skalieren von Kontrollgitterpunkten verwendet.
	float TranslationCenterV;

	//! Horizontale Ebenekoordinate der linken oberen Ecke der Selektionsbox.
	float LeftCornerU;
	//! Vertikale Ebenekoordinate der linken oberen Ecke der Selektionsbox.
	float LeftCornerV;

	//! Horizontale Ebenekoordinate der rechten unteren Ecke der Selektionsbox.
	float RightCornerU;
	//! Vertikale Ebenekoordinate der rechten unteren Ecke der Selektionsbox.
	float RightCornerV;

	//! Gitterzeile des Kontrollpunkts unter dem Mauszeiger.
	int ControlPointRow;
	//! Gitterspalte des Kontrollpunkts unter dem Mauszeiger.
	int ControlPointCol;

	//! Deaktivierter Standardkonstruktor.
	CBSplineWrapperEffect();
	//! Deaktivierter Zuweisungsoperator.
	CBSplineWrapperEffect& operator = (const CBSplineWrapperEffect& source);

};

#endif




