/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.

 	This file is part of GL Effects Framework.

    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CBSplineWrapperEffect.cpp
 *
 *  @brief
 *	Implementation der Klasse \b CBSplineWrapperEffect.
 *
 *  @author Christoph Brzozowski
 *
 */

//! Schwellwert fr das Greifen von Kontrollpunkten.
#define PICK_THERSHOLD 0.03f
//! Schrittweite des Framecursors beim Einfgen von Keyframes.
#define CURSOR_STEP 0.25f

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

// Qt-Headerdateien
#include <qtimer.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qspinbox.h>
#include <qslider.h>
#include <qdialog.h>
#include <qlayout.h>
#include <qgrid.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qvgroupbox.h>
#include <qpoint.h>
#include <qrect.h>
#include <qlabel.h>
#include <qsizepolicy.h>
#include <qpopupmenu.h>
#include <qtooltip.h>
#include <qmessagebox.h>
#include <qtable.h>
#include "QKeyFrameBar.h"
#include "QUnclosableDialog.h"

// Lokale Headerdateien
#include <math.h>
#include "CGrid.h"
#include "CGridPainter.h"
#include "CMeshBuilder.h"
#include "CVector.h"
#include "CMatrix.h"
#include "CLinearDeformer.h"
#include "CBSplineDeformer.h"

// Effektheaderdatei
#include "CBSplineWrapperEffect.h"

/*!
 * Der Konstruktor erzugt eine Instanz der Klasse \b CBSplineWrapper.
 * Die Struktur \b effectInfo wird zunchst initialisert, der Effekt untersttzt zwei Bildlisten.
 * Alle Zeigerreferenzen werden auf NULL gesetzt.
 * Danach werden die Deformationsalgorithmen-Klassen erzegut und in die Algorithmenliste aufgenommen.
 * Anschlieend werden alle Steuerelemente verborgen.
 *
 * @param parent Referenz auf das Elternfenster.
 *
 **/

CBSplineWrapperEffect::CBSplineWrapperEffect(GLFramework* parent):GLEffect(parent)
{

	// Interne Variablen initialisieren.
	ControlMesh = NULL;
	ImageMesh = NULL;
	TexelMesh = NULL;
	TexelMeshA = NULL;
	TexelMeshB = NULL;
	ControlPointColorGrid=NULL;
	ActiveDeformer = NULL;

	// Keyframe-Timer erzeugen
	tmrEffectTimer = new QTimer();

	// Deformer-Klassen erzeugen und in die Deformer-Liste aufnehmen
  	Deformers.insert(new CLinearDeformer<GLfloat>);
	Deformers.insert(new CBSplineDeformer<GLfloat>);

	// Effekt-Informationsstruktur initialisieren
	effectInfo->effectName = "B-Spline-Image-Wrapper";
	effectInfo->author = "Christoph Brzozowski";
	effectInfo->shortDescription = "Allows the deformation and morphing of an image(s) via B-spline interpolation.";
	effectInfo->versionMajor = 1;
	effectInfo->version = 0;
	effectInfo->requiredImageLists = 2;
	effectInfo->needNewTextures = true;
	effectInfo->needRawTextures = false;
	effectInfo->needStreaming = false;
	effectInfo->fileDialogNames[0] = "Choose source image sequence...";
	effectInfo->fileDialogNames[1] = "Choose destination image sequence...";

	// Effekt initialisieren (muss aufgerufen werden, wg. Event-Probleme unter LINUX)
	// Initialize ist absichtlich so angelegt, dass es mehrfach aufgerufen werden kann.
	initialize();

	// Kein automatisches Erzeugen eines Toolfensters.
	createControlPanel(false);

	// Maustracking aktivieren. Wird fr das Verschieben der Kontrollpunkte bentigt.
	forceMouseTracking(true);

	// Steuerelemente verbergen - Achtung!!! dies ist notwendig...
	hideControls();

}

/*
 * Der Destruktor stoppt zunchst den Animationstimer und zerstrt ihn.
 * Danach werden alle dynamisch erzeugten Objekte freigegeben.
 * Weiteres siehe Code...
 *
 * \remarks
 * Es werden nur bestimmte QT-Steuerelemente gezielt freigegeben, da
 * diese ihre Shne automatisch freigeben.
 *
 */
CBSplineWrapperEffect::~CBSplineWrapperEffect()
{

	// Timer anhalten und freigeben
	tmrEffectTimer->stop();
	delete tmrEffectTimer;

	// Deformer-Klassen freigeben
	for (int i=0 ; i<Deformers.count() ; i++)
	{
		delete Deformers[i];
	};

	// Dynamisch alloziierte Klassen freigeben.
	if (ControlMesh!=NULL) delete ControlMesh;
 	if (ImageMesh!=NULL) delete ImageMesh;
	if (TexelMesh!=NULL) delete TexelMesh;
	if (TexelMeshA!=NULL) delete TexelMeshA;
	if (TexelMeshB!=NULL) delete TexelMeshB;
	if (ControlPointColorGrid!=NULL) delete ControlPointColorGrid;

	// QT-Hauptsteuerelemente freigeben
	delete dlgToolWindow;
	delete controlPanel;

}

/*! Die Methode \b initialize() initialisiert den Effekt mit Standardwerten.
 *  Auch die GUI-Elemente werden mit Werten vorbelegt.
 *  Die Methode erzeugt alle Gitter und ntigen dynamischen Strukturen.
 */
void CBSplineWrapperEffect::initialize()
{

	// --- Gitterfarben initialisieren ----------------------------------

	ImageMeshColor.c[0] = 0.0;					// R | Imagemesh-Farbe
	ImageMeshColor.c[1] = 1.0;					// G | ist ein transparentes
	ImageMeshColor.c[2] = 0.0;					// B | Grn
	ImageMeshColor.c[3] = 0.25;         // A |

	ControlMeshColor.c[0] = 1.0;				// R | Controlmesh-Farbe ist
	ControlMeshColor.c[1] = 0.0;				// G | ein deckendes
	ControlMeshColor.c[2] = 0.0;				// B | Rot.
	ControlMeshColor.c[3] = 1.0;				// A |

	ControlMeshVertexColor.c[0] = 0.0;	// R | Controlmeshvertex-Farbe ist
	ControlMeshVertexColor.c[1] = 1.0;  // G | ein deckendes
	ControlMeshVertexColor.c[2] = 1.0;  // B | helles Blau.
	ControlMeshVertexColor.c[3] = 1.0;  // A |

	SelectedControlMeshVertexColor.c[0] = 1.0;	// R | Controlmeshvertex-Farbe
	SelectedControlMeshVertexColor.c[1] = 1.0;  // G | ist ein deckendes
	SelectedControlMeshVertexColor.c[2] = 0.0;  // B | Gelb.
	SelectedControlMeshVertexColor.c[3] = 1.0;  // A |

	// --- Gitterbindungen auflsen ----------------------------------------------

	resolveGridBindings();

	// --- Neue Gitter erzeugen mit den Ausmaen 2x2 ------------------------------

	// Alte Gitter zunchst freigeben
	if (ImageMesh!=NULL) delete ImageMesh;
	if (ControlMesh!=NULL) delete ControlMesh;
	if (TexelMesh!=NULL) delete TexelMesh;
	if (TexelMeshA!=NULL) delete TexelMeshA;
	if (TexelMeshB!=NULL) delete TexelMeshB;
	if (ControlPointColorGrid!=NULL) delete ControlPointColorGrid;

	// Gitter erzeugen
	buildRegularMesh(ControlMesh,-1.0,-1.0,2.0,2.0,2,2);
	buildRegularMesh(ImageMesh,-1.0,-1.0,2.0,2.0,2,2);
	buildRegularMesh(TexelMesh,0.0,0.0,1.0,1.0,2,2);
	TexelMeshA = new CGrid< CVector<2,GLfloat> >(2,2);
	TexelMeshB = new CGrid< CVector<2,GLfloat> >(2,2);

	// Farbgitter fr Kontrollpunkte erzeugen und zurcksetzen.
	ControlPointColorGrid = new CGrid< CVector<4, GLfloat> >();
	ControlPointColorGrid->resize(ControlMesh->width(),ControlMesh->height());
	ControlPointColorGrid->clear(ControlMeshVertexColor);

	// Liste mit selektierten Kontrolpunkten zurcksetzen
	SelectedControlPoints.clear();

	// --- Meshpainter-Objekt initialisieren -----------------------------------

	ControlMeshPainter.pointSize = 4.0;
	ControlMeshPainter.setVertexGrid(ControlMesh,2,GL_FLOAT);
	ControlMeshPainter.setColorGrid(ControlPointColorGrid,4,GL_FLOAT);
	ControlMeshPainter.useVertexGrid = true;

	ImageMeshPainter.setVertexGrid(ImageMesh,2,GL_FLOAT);
	ImageMeshPainter.setTexelGrid(TexelMesh,2,GL_FLOAT);
	ImageMeshPainter.useVertexGrid = true;
	ImageMeshPainter.useTexelGrid = true;

	// --- Keyframe-Liste zurcksetzen ----------------------------------

	KeyFrameList.clear();

	// --- Die Initialzustnde des Effekts setzen -----------------------

	TranslationMode = tmMove;
	TranslationState = tsNothing;
	DeformationMode = dmLinear;
	EffectMode = emKeyFrame;

	// --- Den initalen Zeichenmodus des Effekts setzen -----------------

	PaintControlMesh = true;
	PaintControlMeshVertices = true;
	PaintImage = true;
	PaintImageMesh = false;

	// --- Das Sperren der Randgitterpunkte deaktivieren ----------------

	LockBorder = false;

	// --- Linearen Deformer als aktiven Deformer setzen ----------------

	ActiveDeformer = Deformers[0];
	ActiveDeformer->setSourceGrid(ImageMesh);
	ActiveDeformer->setControlGrid(ControlMesh);
	ActiveDeformer->setDestinationGrid(ImageMesh);

	// --- Splineordnung initialisieren ---------------------------------

	CurveOrder = 2;
	((CBSplineDeformer<GLfloat>*) Deformers[1])->setOrder(CurveOrder);

	// --- Die zu bearbeitende Knotenmatrix setzen ----------------------

	ActiveKnotMatrix = kmRow;

	// --- Morphingparameter initialisieren -----------------------------

	MorphingTimeIndex = 0;
	MorphingEditState = esSource;

	// --- Sichtpyramide initialisieren ---------------------------------

	viewDistance = 3;
	nearClipPlane = 1;
	farClipPlane = 30;
	fovAngle = 60.0;

	currentDepthCoord = 0.0;

	// --- Animationseinstellungen initialsieren ------------------------

	AnimationMode = amOnce;
	AnimationSpeed = 10;
	AnimationDirection = +1;
	tmrEffectTimer->stop();

	// --- Den initialen OpenGL status setzen ---------------------------

	// Schwarz als Lschfarbe.
	glClearColor(0.0,0.0,0.0,1.0);

	// Alphablending aktivieren und Blendingfunktion setzen
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);

	// Flatshading aktivieren.
	glShadeModel(GL_FLAT);

	// Z-Buffer deaktivieren.
	glDisable(GL_DEPTH_TEST);

	// Beleuchtung deaktivieren.
	glDisable(GL_LIGHTING);

	// Das Normalisieren der Normalenvektoren deaktivieren.
	glDisable(GL_NORMALIZE);

	// Antialiasing von Punkten aktivieren.
	glEnable(GL_POINT_SMOOTH);

	// Backfaceculling deaktivieren.
	glDisable(GL_CULL_FACE);

	// Dithering aktivieren.
	glEnable(GL_DITHER);

	// Identittsmatrix laden
	glLoadIdentity();

	// --- OpenGL Texturparamteter initialisieren -----------------------

	// 2-Dimensionales Texturing aktivieren
	glEnable(GL_TEXTURE_2D);

	// Textur-Blendingmodus setzen
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	// Textur-Pixelformat setzen
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	// Texturen erzeugen
	glGenTextures(1, &texture);
	glBindTexture(GL_TEXTURE_2D, texture);

	// Textur-Wrapping-Parameter setzen
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);

	// Textur-Filtering-Parameter setzen, Bi-lineare Filterung aktivieren.
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);

	// Perspektivische Korrektur aktivieren.
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

	// Erstes Frame aus dem Framework laden.
	// texImage1 = parentFrame->fetchImage(0,0);

}

/*! Die Methode \b createControls() erzeugt die vom Effekt zur Verfgung
 *  gestellten Steuerelemente.
 *  Sie erzeugt ein Panel in der rechten Leiste des Frameworks, welches
 *  Checkboxen zum Konfigurieren des Zeichenmodus des Effekts beinhaltet
 *  und ein Panel zum einstellen der Animationsabspielparameter.
 *  Ferner erzegut sie ein Top-Level-Toolfenster, welches
 *  Steuerelemente zur Einstellung der Bearbeitungsmodi, der Gitter-, B-Spline-
 *  und Keyframe-Parameter bereitstellt.
 *  Die Steuerelemente werden anschlieend mit Standardwerten vorbelegt.
 *
 */
void CBSplineWrapperEffect::createControls(QWidget* parentWindow)
{

	// --- Create additional top level tool window -----------------------

	dlgToolWindow = new QUnclosableDialog(0,0,false,
											Qt::WStyle_Customize |
											Qt::WStyle_NormalBorder |
											Qt::WStyle_Title |
											Qt::WStyle_Tool |
											Qt::WStyle_StaysOnTop |
											Qt::WStyle_Minimize);
//											Qt::WStyle_Dialog |


	dlgToolWindow->setCaption("Control panel");

	// --- Create tool window widgets ------------------------------------

	// create tool window layout
	layToolWindow = new QGridLayout(dlgToolWindow,1,1);

	// create tool window main panel
	pnlToolWindow = new QVBox(dlgToolWindow);
	pnlToolWindow->setSpacing(2);
	layToolWindow->addWidget(pnlToolWindow,0,0);

	// create effect mode panel
	pnlMode = new QHBox(pnlToolWindow);

	// create mode buttons
	btnTranslationMode = new QPushButton("Move",pnlMode);
	btnDeformationMode = new QPushButton("Linear",pnlMode);
	btnEffectMode = new QPushButton("Key frame",pnlMode);

	QToolTip::add(btnTranslationMode, "Selects the control point's translation mode");
	QToolTip::add(btnDeformationMode, "Selects the image mesh's deformation mode");
	QToolTip::add(btnEffectMode, "Selects the effect's mode");

	// --- Create popup menus --------------------------------------------

	popTranslationMode = new QPopupMenu(btnTranslationMode);
	popTranslationMode->insertItem("&Move",tmMove,0);
	popTranslationMode->insertItem("&Rotate",tmRotate,1);
	popTranslationMode->insertItem("&Scale",tmScale,2);
	btnTranslationMode->setPopup(popTranslationMode);

	popDeformationMode = new QPopupMenu(btnDeformationMode);
	popDeformationMode->insertItem("&Linear",dmLinear,0);
	popDeformationMode->insertItem("&B-Spline",dmBSpline,1);
	btnDeformationMode->setPopup(popDeformationMode);

	popEffectMode = new QPopupMenu(btnEffectMode);
	popEffectMode->insertItem("&Key frame",emKeyFrame,0);
	popEffectMode->insertItem("&Morphing",emMorphing,1);
	btnEffectMode->setPopup(popEffectMode);

	// create mesh parameter panel
	pnlGridParam = new QHBox(pnlToolWindow);
	pnlGridParam->setSpacing(2);

	// create b-spline order group
	grpOrder = new QVGroupBox("Spline",pnlGridParam);
	lblOrder = new QLabel("Order:",grpOrder);
	spnOrder = new QSpinBox(2,5,1,grpOrder);
	lblOrder->setBuddy(spnOrder);

	QToolTip::add(spnOrder, "Sets the B-spline curve order");

	// create mesh resolution group
	grpResolution = new QVGroupBox("Mesh resolution",pnlGridParam);

	// create control mesh resolution panel
	lblControlMesh = new QLabel("Control mesh:",grpResolution);
	hbxControlMeshRes = new QHBox(grpResolution);
	hbxControlMeshRes->setSpacing(2);

	lblHorResControlMesh = new QLabel("U:", hbxControlMeshRes);
	lblHorResControlMesh->setAlignment(AlignRight | AlignVCenter | ExpandTabs );
	spnHResCMesh = new QSpinBox(2,16,1,hbxControlMeshRes);
	QToolTip::add(spnHResCMesh, "Sets the control mesh's horizontal resolution");

	lblVerResControlMesh = new QLabel("V:", hbxControlMeshRes);
	lblVerResControlMesh->setAlignment(AlignRight | AlignVCenter | ExpandTabs );
	spnVResCMesh = new QSpinBox(2,16,1,hbxControlMeshRes);
	QToolTip::add(spnVResCMesh, "Sets the control mesh's vertical resolution");

	// vertex lock checkbox
	cbxLockBorder = new QCheckBox("Lock border vertices",grpResolution);
	QToolTip::add(cbxLockBorder,"Locks the movement of the control mesh border vertices");
	btnResetMesh = new QPushButton("Reset",grpResolution);
	btnResetMesh->setMaximumWidth(76);
	QToolTip::add(btnResetMesh,"Sets the control mesh to a regular one");

	// create image mesh resolution panel
	lblImageMesh= new QLabel("ImageMesh:",grpResolution);
	hbxImageMeshRes = new QHBox(grpResolution);
	hbxImageMeshRes->setSpacing(2);

	lblHorResImageMesh = new QLabel("U:", hbxImageMeshRes);
	lblHorResImageMesh->setAlignment(AlignRight | AlignVCenter | ExpandTabs );
	spnHResIMesh = new QSpinBox(2,256,1,hbxImageMeshRes);
	QToolTip::add(spnHResIMesh, "Sets the image mesh's horizontal resolution");

	lblVerResImageMesh = new QLabel("U:", hbxImageMeshRes);
	lblVerResImageMesh->setAlignment(AlignRight | AlignVCenter | ExpandTabs );
	spnVResIMesh = new QSpinBox(2,256,1,hbxImageMeshRes);
	QToolTip::add(spnVResIMesh, "Sets the image mesh's vertical resolution");

	// create node matrix group
	grpNodeMatrix = new QVGroupBox("Knot matrix",pnlGridParam);

	tabKnotMatrix = new QTable(grpNodeMatrix);
  tabKnotMatrix->setLeftMargin( 0 );
  tabKnotMatrix->verticalHeader()->hide();
	tabKnotMatrix->setTopMargin(0);
	tabKnotMatrix->horizontalHeader()->hide();
	QToolTip::add(tabKnotMatrix,"Double click to edit the values");

	pnlKnotMatrixNavigation = new QHBox(grpNodeMatrix);
	btnMatrixType = new QPushButton("Row matrix",pnlKnotMatrixNavigation);
	QToolTip::add(btnMatrixType,"Selects the matrix type");

	popMatrixType = new QPopupMenu(btnMatrixType);
	popMatrixType->insertItem("&Row matrix",kmRow,0);
	popMatrixType->insertItem("&Col matrix",kmCol,1);
	btnMatrixType->setPopup(popMatrixType);

	btnResetKnotMatrix = new QPushButton("Reset",pnlKnotMatrixNavigation);
	btnResetKnotMatrix->setMaximumWidth(76);
	QToolTip::add(btnResetKnotMatrix,"Makes both knot matrices uniform again");

	// create frame navigation panel
	pnlFrameNavigation = new QHBox(pnlToolWindow);
	pnlFrameNavigation->setSpacing(0);
	pnlFrameNavigation->setMaximumHeight(28);

	// create frame navigation buttons
	btnZero = new QPushButton("Zero",pnlFrameNavigation);
	btnFirst = new QPushButton("First",pnlFrameNavigation);
	btnPrevious = new QPushButton("Previous",pnlFrameNavigation);
	btnNext = new QPushButton("Next", pnlFrameNavigation);
	btnLast = new QPushButton("Last", pnlFrameNavigation);
	btnInsert = new QPushButton("Insert",pnlFrameNavigation);
	btnDelete = new QPushButton("Delete",pnlFrameNavigation);
	btnClear = new QPushButton("Clear",pnlFrameNavigation);

  QToolTip::add(btnZero,"Sets the frame cursor's position to zero.");
	QToolTip::add(btnFirst, "Jumps to the list's first key frame");
	QToolTip::add(btnPrevious, "Jumps to previous key frame");
	QToolTip::add(btnNext, "Jumps to next key frame");
	QToolTip::add(btnLast, "Jumps to the list's last key frame");
	QToolTip::add(btnDelete,"Deletes the selected key frame");
	QToolTip::add(btnInsert,"Inserts a new key frame at the frame cursor position");
	QToolTip::add(btnClear,"Deletes all key frames");

	// create morphing navigation panel
	pnlMorphingNavigation = new QHBox(pnlToolWindow);
	pnlMorphingNavigation->setSpacing(0);
	pnlMorphingNavigation->setMaximumHeight(28);

	// create morphing navigation buttons
	btnMorphToSource = new QPushButton("To Source",pnlMorphingNavigation);
	QToolTip::add(btnMorphToSource,"Copies the current control mesh to the source frame");
	btnMorphFromSource = new QPushButton("From Source",pnlMorphingNavigation);
	QToolTip::add(btnMorphFromSource,"Retrieves the control mesh from the source frame");
	btnMorphToDest = new QPushButton("To Destination",pnlMorphingNavigation);
	QToolTip::add(btnMorphToDest,"Copies the current control mesh to the destination frame");
	btnMorphFromDest = new QPushButton("From Destination",pnlMorphingNavigation);
	QToolTip::add(btnMorphFromDest,"Retrieves the control mesh from the destination frame");

	// create key frame bar panel
	pnlFrameBar = new QHBox(pnlToolWindow);
	pnlFrameBar->setMaximumHeight(28);

	// create key frame bar grid
	pnlFrameGrid = new QGrid(2,pnlFrameBar);
	pnlFrameGrid->setMaximumWidth(32);

	// create key frame bar buttons
	btnZoomIn = new QPushButton("+",pnlFrameGrid);
	btnZoomIn->setMaximumSize(14,14);
	QToolTip::add(btnZoomIn,"Zoom in");

	btnZoomOut = new QPushButton("-",pnlFrameGrid);
	btnZoomOut->setMaximumSize(14,14);
	QToolTip::add(btnZoomOut,"Zoom out");

	btnScrollLeft = new QPushButton("<",pnlFrameGrid);
	btnScrollLeft->setMaximumSize(14,14);
	QToolTip::add(btnScrollLeft,"Scroll left");

	btnScrollRight = new QPushButton(">",pnlFrameGrid);
	btnScrollRight->setMaximumSize(14,14);
	QToolTip::add(btnScrollRight,"Scroll right");

	// create key frame time bar
	usrKeyFrameBar = new QKeyFrameBar(pnlFrameBar);

	// bind key frame time bar to the keyframe list
	usrKeyFrameBar->setFrameList(&KeyFrameList);

	// --- Create paint mode panel ---------------------------------------

	controlPanel = new QVBox(parentWindow);

	grpPaintMode = new QVGroupBox("Paint mode",controlPanel);
	cbxPaintImage = new QCheckBox("Show image",grpPaintMode);
	QToolTip::add(cbxPaintImage,"Toggles the image's paint mode");
	cbxPaintImageMesh = new QCheckBox("Show mesh",grpPaintMode);
	QToolTip::add(cbxPaintImageMesh,"Toggles the image mesh's paint mode");
	cbxPaintControlMesh = new QCheckBox("Show control mesh",grpPaintMode);
	QToolTip::add(cbxPaintControlMesh,"Toggles the control mesh's paint mode");
	cbxPaintControlMeshVertices = new QCheckBox("Show control points",grpPaintMode);
	QToolTip::add(cbxPaintControlMeshVertices,"Toggles the control mesh vertices' paint mode");

	// --- Create animation control panel --------------------------------

  grpAnimationOptions = new QVGroupBox("Animation",controlPanel);
	btnAnimationMode = new QPushButton("Once",grpAnimationOptions);
	sldAnimationSpeed = new QSlider(0, 100, 10, 10, QSlider::Horizontal,grpAnimationOptions);
	sldAnimationSpeed->setTracking(true);
	sldAnimationSpeed->setTickmarks(QSlider::Below);
	sldAnimationSpeed->setTickInterval(10);

	QToolTip::add(btnAnimationMode, "Sets the animation mode");
	QToolTip::add(sldAnimationSpeed,"Sets the animation speed");

	hbxFrameLoop = new QHBox(grpAnimationOptions);
	lblFrameLoopCount = new QLabel("Frame loops: ",hbxFrameLoop);
	spnFrameLoop = new QSpinBox(1,100,1,hbxFrameLoop);
	QToolTip::add(spnFrameLoop,"Sets the number of frame sequence loops");

	// --- Create animation mode popup menu ------------------------------

	popAnimationMode = new QPopupMenu(btnAnimationMode);
	popAnimationMode->insertItem("&Once",amOnce,0);
	popAnimationMode->insertItem("&Loop",amLoop,1);
	popAnimationMode->insertItem("&Ping-Pong",amPingPong,2);
	btnAnimationMode->setPopup(popAnimationMode);

	// --- Connect signals to slots --------------------------------------

	// Mode popup menus
	connect(popTranslationMode,SIGNAL(activated(int)),this,SLOT(setTranslationMode(int)));
	connect(popDeformationMode,SIGNAL(activated(int)),this,SLOT(setDeformationMode(int)));
	connect(popEffectMode,SIGNAL(activated(int)),this,SLOT(setEffectMode(int)));
	connect(popAnimationMode,SIGNAL(activated(int)),this,SLOT(setAnimationMode(int)));

	// Mesh resolution widgets
	connect(spnHResCMesh,SIGNAL(valueChanged(int)),this,SLOT(setControlMeshHorizontalResolution(int)));
	connect(spnVResCMesh,SIGNAL(valueChanged(int)),this,SLOT(setControlMeshVerticalResolution(int)));

	connect(spnHResIMesh,SIGNAL(valueChanged(int)),this,SLOT(setImageMeshHorizontalResolution(int)));
	connect(spnVResIMesh,SIGNAL(valueChanged(int)),this,SLOT(setImageMeshVerticalResolution(int)));

	// Vertex lock check box
	connect(cbxLockBorder,SIGNAL(toggled(bool)),this,SLOT(setBorderLock(bool)));

	// Control mesh reset button
	connect(btnResetMesh,SIGNAL(clicked()),this,SLOT(resetControlMesh()));

	// Knot matrix table
	connect(tabKnotMatrix,SIGNAL(valueChanged(int,int)),this,SLOT(updateKnotMatrixValue(int,int)));

	// Knot matrix reset button
	connect(btnResetKnotMatrix,SIGNAL(clicked()),this,SLOT(resetKnotMatrix()));

	// Knot matrix mode button
	connect(popMatrixType,SIGNAL(activated(int)),this,SLOT(setActiveKnotMatrix(int)));

	// Spline order spin box
	connect(spnOrder,SIGNAL(valueChanged(int)),this,SLOT(setSplineOrder(int)));

	// Frame navigation buttons
	connect(btnZero,SIGNAL(clicked()),this,SLOT(gotoZero()));
	connect(btnInsert,SIGNAL(clicked()),this,SLOT(insertKeyFrame()));
	connect(btnDelete,SIGNAL(clicked()),this,SLOT(deleteKeyFrame()));
	connect(btnFirst,SIGNAL(clicked()),this,SLOT(gotoFirstKeyFrame()));
	connect(btnLast ,SIGNAL(clicked()),this,SLOT(gotoLastKeyFrame()));
	connect(btnNext ,SIGNAL(clicked()),this,SLOT(gotoNextKeyFrame()));
	connect(btnPrevious,SIGNAL(clicked()),this,SLOT(gotoPreviousKeyFrame()));
	connect(btnClear,SIGNAL(clicked()),this,SLOT(clearKeyFrames()));
	connect(btnScrollLeft,SIGNAL(clicked()),usrKeyFrameBar,SLOT(scrollLeft()));
	connect(btnScrollRight,SIGNAL(clicked()),usrKeyFrameBar,SLOT(scrollRight()));
	connect(btnZoomIn,SIGNAL(clicked()),usrKeyFrameBar,SLOT(zoomIn()));
	connect(btnZoomOut,SIGNAL(clicked()),usrKeyFrameBar,SLOT(zoomOut()));

	// Morphing navigation buttons

	connect(btnMorphToSource,SIGNAL(clicked()),this,SLOT(controlMeshToSource()));
	connect(btnMorphToDest,SIGNAL(clicked()),this,SLOT(controlMeshToDestination()));
	connect(btnMorphFromSource,SIGNAL(clicked()),this,SLOT(sourceToControlMesh()));
	connect(btnMorphFromDest,SIGNAL(clicked()),this,SLOT(destinationToControlMesh()));

	// Paint mode widgets
	connect(cbxPaintImage,SIGNAL(toggled(bool)),this,SLOT(setPaintImage(bool)));
	connect(cbxPaintImageMesh,SIGNAL(toggled(bool)),this,SLOT(setPaintImageMesh(bool)));
	connect(cbxPaintControlMesh,SIGNAL(toggled(bool)),this,SLOT(setPaintControlMesh(bool)));
	connect(cbxPaintControlMeshVertices,SIGNAL(toggled(bool)),this,SLOT(setPaintControlMeshVertices(bool)));

	// Frame bar slots
	connect(usrKeyFrameBar,SIGNAL(interpolateAt(float)),this,SLOT(interpolate(float)));
	connect(usrKeyFrameBar,SIGNAL(finishInterpolate()),this,SLOT(finishInterpolate()));

	// Animation speed slider
	connect(sldAnimationSpeed,SIGNAL(valueChanged(int)),this,SLOT(setAnimationSpeed(int)));

	// Key frame timer
	connect(tmrEffectTimer,SIGNAL(timeout()),this,SLOT(animate()));

	// --- setup initial GUI state ---------------------------------------
	initializeGUI();

}

/*! Die Methode \b initializeGUI() belegt alle Steuerelemente des Effekts
 *  mit Standardwerten.
 *  Der Effekt wird in den Keyframemodus zurckgesetzt. Der Manipulationsmodus
 *  wird auf "Move" gesetzt. Der Lineare Deformationsalgorithmus wird aktiviert.
 *  Alle Keyframes werden gelscht.
 *  Die Animationseinstellungen werden zurckgesetzt.
 *  Die Zeichenmoduseinstellungen ebenfalls. Die Gitter werden alle 2 x 2
 *  dimensioniert. Die Splineordnung wird ebenfalls auf 2 zurckgesetzt.
 */
void CBSplineWrapperEffect::initializeGUI()
{

	// --- Set initial GUI state ------------------------------

	// Set paintmode checkboxes
	cbxPaintImage->setChecked(true);
	cbxPaintImageMesh->setChecked(false);
	cbxPaintControlMesh->setChecked(true);
	cbxPaintControlMeshVertices->setChecked(true);

	// Set mode button captions
	btnTranslationMode->setText("Move");
	btnDeformationMode->setText("Linear");
	btnEffectMode->setText("Key frame");

	// Hide morphing panel, show keyframe panel
	pnlMorphingNavigation->hide();
	pnlFrameNavigation->show();

	// Enable key frame movement
	usrKeyFrameBar->setFrameMoving(true);

	// Set b-spline order
	spnOrder->setValue(2);
	grpOrder->setEnabled(true);
	spnOrder->setEnabled(false);

	// Set grid resolution
	spnHResCMesh->setMinValue(2);
	spnHResCMesh->setValue(2);
	spnVResCMesh->setMinValue(2);
	spnVResCMesh->setValue(2);
	spnHResIMesh->setValue(2);
	spnVResIMesh->setValue(2);

	enableControlMeshResolutionPanel(true);

	// Disable border lock
	cbxLockBorder->setChecked(false);

	// Set node matrix
	enableKnotMatrixPanel(false);

	btnMatrixType->setText("Row matrix");

	// Set animation speed and animation mode buttons
	sldAnimationSpeed->setValue(10);
	btnAnimationMode->setText("Once");
	spnFrameLoop->setValue(1);

	// Set key frame bar
	usrKeyFrameBar->setDisplayRange(-2.0,2.0);
	usrKeyFrameBar->setCursorPosition(0);
  usrKeyFrameBar->centerView();

};

// Hides the effect
void CBSplineWrapperEffect::hide()
{
	// Stop any animation playing
	stop();

	// Call base class version of method
	GLEffect::hide();

	// Hide control window
	dlgToolWindow->hide();

};

// Shows the effect
void CBSplineWrapperEffect::show()
{

	// Call base class version of method
	GLEffect::show();

	// Show top level tool window
	QPoint tmpPoint(0,0);
	tmpPoint = glView->mapToGlobal(tmpPoint);
	dlgToolWindow->move(tmpPoint.x(),tmpPoint.y());
	dlgToolWindow->show();
	dlgToolWindow->raise();

};

/*! Die Methode \b play() startet den Abspielvorgang im Morphing, oder Keyframemodus.
 *  Dieser wird nur dann gestartet, wenn sich mehr als 1 Keyframe in der Liste befindet.
 *  Die Methode startet den Animationstimer, der die Animation in regelmigen Abstnden aktualisiert.
 *  Das Toolfenster des Effekts wird durch die Methode versteckt.
 */
void CBSplineWrapperEffect::play()
{

	// Prfen, ob mindestens 2 keyframes vorhanden sind
	if (KeyFrameList.count()>1)
	{

		// Hide tool window
		dlgToolWindow->hide();

		// Start timer
		tmrEffectTimer->start(0);

		update();

	};

}

/*! Die Methode \b stop() hlt eine laufende Animation an.
 *  Das Effekttoolfenster wird wieder eingeblendet, der Animationstimer
 *  gestoppt.
 */
void CBSplineWrapperEffect::stop()
{

	// Reset animation direction
	AnimationDirection = 1;

	// Stop timer
	tmrEffectTimer->stop();

	// Show tool window
	dlgToolWindow->show();

	// Reset translation state
	TranslationState = tsNothing;

	// Restore morphing editing state
	if (EffectMode == emMorphing)
	{
		sourceToControlMesh();
	};

	update();

}

/*! Die Methode \b reset() setzt den Effekt in seinen Ausgangszustand zurck.
 *  Eine mgliche Animation wird zunchst angehalten.
 *  Danach werden alle erzeugten OpenGL-Texturobjekte gelscht.
 *  Der aktive Deformer wird von den Gittern getrennt. Die Gitter werden
 *  von ihren Painte-Objekten ebenfalls getrennt.
 *  Danach werden \b initialize() und \b initializeGUI() aufgerufen, um
 *  die Standardzustnde des Effekts zu initialisieren.
 */
void CBSplineWrapperEffect::reset()
{

	// Stop animation
	stop();

	// Delete Textures
	glDeleteTextures(1,&texture);

	// Now we simply recall initialize
	initialize();

	// And reintialize the GUI
	initializeGUI();

	// Update view
	update();

}


/*! Die Methode \b stop() unterbricht eine laufende Animation.
 *  Das Effekttoolfenster wird dabei nicht wieder eingeblendet, es wird lediglich der Animationstimer
 *  gestoppt.
 */
void CBSplineWrapperEffect::pause()
{

	// Stop timer
	tmrEffectTimer->stop();

}

/*! Die Methode \b render() rendert den Effekt. Je nach Konfiguration
 *  der Darstellungsmodi wird zunchst das texturierte Bildgittergezeichnet,
 *  danach das Bildgitter im Wireframemodus, dann das Kontrollgitter
 *  und anschlieend die Kontrollgitterpunkte.
 *  Die Methode zeichnet ebenfalls ein Fadenkreuz und die Punkteselektionsbox.
 *
 *  Im Morphingmodus wird das texturierte Bildgitter zweimal gezeichnet.
 *  Einmal mit der aktuellen Textur der Quellbildsequenz, einmal mit der
 *  Textur der aktuellen Textur der Zielbildsequenz. Dabei variiert jedesmal
 *  der Alphablending wert, in abhngigkeit des Morphingzeitindex.
 *  Fr das Rendern des Quellgitters wird \b TexelmeshA verwendet, fr
 *  das Rendern des Zielgitters wird \b TexelmeshB verwendet.
 *
 *  Die Methode \b render() ruft jedesmal ebenfalls die Methode \b deform()
 *  des gerade aktiven Deformationsalgorithmus auf. Im Morphingmodus werden
 *  dabei jedesmal die Texelkoordinaten neuberechnet, so da die Verzerrung
 *  keinerlei Auswirkungen auf das Bild hat.
 *
 *  Das Zeichnen der Gitter wird mit Hilfe des Objekts \b CGridPainter
 *  bewerkstelligt.
 */
void CBSplineWrapperEffect::render()
{

	// Perform deformation

	if (ActiveDeformer!=NULL)
	{

		ActiveDeformer->deform();

		// Check if effect is in morphing mode and no interpolation is beeing performed
		if ((EffectMode == emMorphing) && ( (TranslationState!=tsInterpolating) || (TranslationState!=tsNothing)))
		{
			// recalculate texel grid
			recalculateTexelCoordinates(*ImageMesh, *TexelMesh);
		};

	}

	// Clear frame buffer and z-buffer
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);

	glEnable(GL_TEXTURE_2D);

	// Render image mesh if activated
	if (PaintImage)
	{

		switch (EffectMode)
		{
			case emKeyFrame:
			{

				// hook up the current texture
				if(texImage1!=0)
				glTexImage2D(GL_TEXTURE_2D,0,4,texImage1->width(), texImage1->height(),
									   0, GL_RGBA, GL_UNSIGNED_BYTE, texImage1->bits());

				glBindTexture(GL_TEXTURE_2D, texture);

				glColor4f(1.0,1.0,1.0,1.0);
				ImageMeshPainter.paintMode = CGridPainter::pmSolid;
				ImageMeshPainter.render();

			}
			break;

			case emMorphing:
			{

				if (TranslationState==tsInterpolating)
				{

					// hook up the current source texture
					if(texImage1!=0)
					glTexImage2D(GL_TEXTURE_2D,0,4,texImage1->width(), texImage1->height(),
											 0, GL_RGBA, GL_UNSIGNED_BYTE, texImage1->bits());

					glBindTexture(GL_TEXTURE_2D, texture);

					glColor4f(1.0,1.0,1.0,1.0-MorphingTimeIndex);

					ImageMeshPainter.setTexelGrid(NULL);
					ImageMeshPainter.setTexelGrid(TexelMeshA,2,GL_FLOAT);
					ImageMeshPainter.paintMode = CGridPainter::pmSolid;
					ImageMeshPainter.render();

					// hook up the current destination texture
					if(texImage2!=0)
					glTexImage2D(GL_TEXTURE_2D,0,4,texImage2->width(), texImage2->height(),
											 0, GL_RGBA, GL_UNSIGNED_BYTE, texImage2->bits());

					glBindTexture(GL_TEXTURE_2D, texture);

					glColor4f(1.0,1.0,1.0,MorphingTimeIndex);

					ImageMeshPainter.setTexelGrid(NULL);
					ImageMeshPainter.setTexelGrid(TexelMeshB,2,GL_FLOAT);
					ImageMeshPainter.paintMode = CGridPainter::pmSolid;
					ImageMeshPainter.render();
				}
				else
				{

					// Set texel mesh to the working texel mesh
					ImageMeshPainter.setTexelGrid(NULL);
					ImageMeshPainter.setTexelGrid(TexelMesh,2,GL_FLOAT);

					// hook up the current texture
					if(texImage1!=0)
					glTexImage2D(GL_TEXTURE_2D,0,4,texImage1->width(), texImage1->height(),
										   0, GL_RGBA, GL_UNSIGNED_BYTE, texImage1->bits());

					glBindTexture(GL_TEXTURE_2D, texture);

					glColor4f(1.0,1.0,1.0,1.0);
					ImageMeshPainter.paintMode = CGridPainter::pmSolid;
					ImageMeshPainter.render();

				};

			}
			break;

		}
	};

	glDisable(GL_TEXTURE_2D);

	// Render image mesh in wire frame mode if activated
	if (PaintImageMesh)
	{
		ImageMeshPainter.setTexelGrid(NULL);
		ImageMeshPainter.setTexelGrid(TexelMesh,2,GL_FLOAT);
		glColor4f(ImageMeshColor.c[0],ImageMeshColor.c[1],ImageMeshColor.c[2],ImageMeshColor.c[3]);
		ImageMeshPainter.paintMode = CGridPainter::pmWireframe;
		ImageMeshPainter.lineWidth = 1.0;
		ImageMeshPainter.render();
	};

	// Render control grid if activated
	if (PaintControlMesh)
	{
		glColor4f(ControlMeshColor.c[0],ControlMeshColor.c[1],ControlMeshColor.c[2],ControlMeshColor.c[3]);
		ControlMeshPainter.paintMode = CGridPainter::pmWireframe;
		ControlMeshPainter.lineWidth = 1.0;
		ControlMeshPainter.useColorGrid = false;
		ControlMeshPainter.render();
	};

	// Render control grid vertices if activated
	if (PaintControlMeshVertices)
	{

		// Paint shadow vertices
		ControlMeshPainter.paintMode = CGridPainter::pmPoints;
		ControlMeshPainter.pointSize = 5.0;
		ControlMeshPainter.useColorGrid = false;

		glTranslatef(0.005f,-0.005f,0.0f);
		glColor4f(0.0f,0.0f,0.0f,0.35f);
		ControlMeshPainter.render();

		// Paint original vertices
		glColor4f(ControlMeshVertexColor.c[0],ControlMeshVertexColor.c[1],ControlMeshVertexColor.c[2],ControlMeshVertexColor.c[3]);
		ControlMeshPainter.paintMode = CGridPainter::pmPoints;
		ControlMeshPainter.pointSize = 5.0;
		ControlMeshPainter.useColorGrid = true;

		glTranslatef(-0.005f,0.005f,0.0f);
		ControlMeshPainter.render();

	};

	// Render a plane cursor, only if animation is not running
	if (!tmrEffectTimer->isActive())
	{
		glColor3f(1.0,1.0,1.0);
		glBegin(GL_LINES);
			glVertex2f(ControlGridPlaneU-0.05,ControlGridPlaneV);
			glVertex2f(ControlGridPlaneU+0.05,ControlGridPlaneV);
			glVertex2f(ControlGridPlaneU,ControlGridPlaneV-0.05);
			glVertex2f(ControlGridPlaneU,ControlGridPlaneV+0.05);
		glEnd();
	};

	// Paint marking box
	if (TranslationState == tsBoxing)
	{
		glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
		glColor4f(1.0f,0.0f,1.0f,0.35f);
		glBegin(GL_QUADS);
			glVertex2f(LeftCornerU,LeftCornerV);
			glVertex2f(RightCornerU,LeftCornerV);
			glVertex2f(RightCornerU,RightCornerV);
			glVertex2f(LeftCornerU,RightCornerV);
		glEnd();

		glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
		glColor4f(1.0f,0.0f,1.0f,1.0f);
		glBegin(GL_QUADS);
			glVertex2f(LeftCornerU,LeftCornerV);
			glVertex2f(RightCornerU,LeftCornerV);
			glVertex2f(RightCornerU,RightCornerV);
			glVertex2f(LeftCornerU,RightCornerV);
		glEnd();

	};

}

/*! Die Methode \b buildRegularMesh() erzeugt mit Hilfe der
 *  Klasse \b CMeshBuilder ein regelmiges rechteckiges Gitter.
 *  Die Parameter \b origX und \b origY geben die linke obere Ecke
 *  des Gitters in Weltkoordinaten an. Die Parameter
 *  \b width und \b height geben die Ausdehnung des Gitters in Weltkooridnaten an.
 *  Die Parameter \b ResH und \b ResV legen schlielich die Auflsung des Gitters fest.
 *  Die Referenz auf das dynamisch erzegute \b CGrid -Objekt wird in \b Mesh abgelegt.
 */
void CBSplineWrapperEffect::buildRegularMesh(CGrid< CVector<2,GLfloat> >* &Mesh, GLfloat origX, GLfloat origY, GLfloat width, GLfloat height, int ResH, int ResV)
{
	// Use a CMeshBuilder-object to create a regular mesh

	CMeshBuilder< 2 , GLfloat > tmpMeshBuilder;

	tmpMeshBuilder.origin.c[0] = origX;
	tmpMeshBuilder.origin.c[1] = origY;
	tmpMeshBuilder.width = width;
	tmpMeshBuilder.height = height;
	tmpMeshBuilder.horResolution = ResH;
	tmpMeshBuilder.verResolution = ResV;

	// Setup the plane spanning vectors

	tmpMeshBuilder.u.c[0]= 1.0;
	tmpMeshBuilder.u.c[1]= 0.0;

	tmpMeshBuilder.v.c[0]= 0.0;
	tmpMeshBuilder.v.c[1]= 1.0;

	// Build mesh
	Mesh = tmpMeshBuilder.build();

};

/*! Die Methode \b makeRegularMesh() verhlt sich analog zur Methode
 *  \b buildRegularMesh(). Sie erzeugt jedoch kein neues \b CGrid-Objekt, sondern
 *  berechnet die Vertexkoordinaten des ber \b Mesh spezifizierten Gitters
 *  neu. Aus diesem Grunde entfllt auch die Angabe der Gitterauflsung.
 *
 *  \pre \b Mesh muss auf ein gltiges \b CGrid- Objekt zeigen.
 */
void CBSplineWrapperEffect::makeRegularMesh(CGrid< CVector<2,GLfloat> >* &Mesh, GLfloat origX, GLfloat origY, GLfloat width, GLfloat height)
{

	for (int y = 0; y<Mesh->height() ; y++)
	{

		for (int x = 0; x<Mesh->width() ; x++)
		{

			// Calculate vertex
			CVector<2,GLfloat> res;
			res.c[0] = ((float) x * width) / (float) (Mesh->width()-1) + origX ;
			res.c[1] = ((float) y * height) / (float) (Mesh->height()-1) + origY;
			Mesh->set(x,y,res);

		};

	};

};

/*! Die Methode \b getPlaneRayIntersection() berechnet den Schnittpunkt zwischen einem
 *  durch \b rayOrigin und \b rayDirection spezifizierten Strahl und einer durch
 *  \b planeOrigin, \b planeU und \b PlaneV parametrisierten Eben.
 *  Das Resultat wird in \b resultU und \b resultV zurckgegeben, d.h. der Schnittpunkt wird in
 *  Ebenenkoordinaten zurckgeliefert.
 *  Die Methode gibt \b true zurck, wenn ein Schnittpunkt existiert, \b false sonst.
 *  In diesem Fall sind \b resultU und \b resultV undefiniert.
 */
bool CBSplineWrapperEffect::getPlaneRayIntersection(CVector<3,float> rayOrigin,
														 CVector<3,float> rayDirection,
														 CVector<3,float> planeOrigin,
														 CVector<3,float> planeU,
														 CVector<3,float> planeV, float &resultU, float &resultV)
{

	// Negate ray direction
	rayDirection.c[0] = -rayDirection.c[0];
	rayDirection.c[1] = -rayDirection.c[1];
	rayDirection.c[2] = -rayDirection.c[2];

	// Calculate relative position between ray and plane
	CVector<3,float> relPos = rayOrigin-planeOrigin;

	// Calculate determinant
	float Det = (planeU % planeV) * rayDirection;

	// Check if determinant is zero which means, that
	// the 3 vectors are coplanar

	if (fabsf(Det) < 1E-8)
	{

		// No intersection
		resultU=0;
		resultV=0;
		return false;

	};

	// Calculate result determinants

	float resDetU = (relPos % planeV) * rayDirection;
	float resDetV = (planeU % relPos) * rayDirection;

	// Calculate plane coordinates

	resultU = resDetU / Det;
	resultV = resDetV / Det;

	return true;

};

// Returns the grid indices of the control point
// which is picked by the mouse.
// The mouse coordinates have to be the ones projected
// onto the control grid plane.

/*! Die Methode getControlPointAtMouse() liefert die Gitterindices des Kontrollunkts zurck,
 *  der sich unter dem Mauszeiger befindet.
 *  Der Parameter \b threshold gibt dabei einen Schwellwert an, der bestimmt wie nah ein Punkt an der Maus sein muss damit
 *  er als angewhlt angesehen wird. Die Parameter \b mx und \b my mssen die bereits auf die Kontrollgitterebene umgerechneten Mauskoordinaten sein.
 *  Die Methode liefert das Ergebnis in \b pntCol und \b pntRow zurck.
 *  Die Methode gibt \b true zurck, falls ein Punkt getroffen wurde, \b false sonst.
 *  In diesem Fall sind \b pntCol und \b pntRow gleich -1.
 *
 *  \remarks
 *  Die Methode bestimmt, ob sich ein Punkt unter dem Mauszeiger befindet,
 *  indem sie zunchst den Punkt ermittelt der die geringste Distanz zum
 *  Mauszeiger hat, und dann prft, ob die Distanz den Schwellwert unterschreitet.
 */
bool CBSplineWrapperEffect::getControlPointAtMouse(float threshold, float mx, float my, int &pntCol, int &pntRow)
{

	float distance; // Stores the distance to the mouse cursor
	int u,v; // Stores the grid index coordinates of the closest point

	u=v=0;

	for (int x = 0 ; x < ControlMesh->width() ; x++)
	{

		for (int y = 0 ; y < ControlMesh->height(); y++)
		{

			CVector<2,float> pntDist;
			pntDist.c[0] = mx - ControlMesh->get(x,y).c[0];
			pntDist.c[1] = my - ControlMesh->get(x,y).c[1];

			if ((x==0) && (y==0))
			{
				distance = pntDist.length();
			}
			else
			if (pntDist.length()<=distance)
			{
				distance = pntDist.length();
				u = x;
				v = y;
			};

		};

	};

	// Check if distance is smaller than the given thershold
	if (distance<=threshold)
	{

		// Point found, return it's grid coordinates
		pntCol = u;
		pntRow = v;
		return true;

	}
	else
	{

		// No point has been found at the give mouse coordinate
		pntCol = -1;
		pntRow = -1;
		return false;

	}

};

/*! Die Methode \b getControlPlaneCoordinates()
 *  projiziert die Mauskoordinaten zurck auf die Kontrollgitterebene.
 *  Das Ergebnis wird in \b ControlGridPlaneU und \b ControlGridPlaneV
 *  abgelegt.
 *
 *  \remarks
 *  Die Routine ermittelt mit Hilfe der Frameworkfunktion
 *  \b get3DMouseCoords() einen Strahl und berechnet seinen
 *  Schnittpunkt mit der Kontrollgitterebene mittels der Methode
 *  \b getPlaneRayIntersection()
 *
 */
void CBSplineWrapperEffect::getControlPlaneCoordinates()
{

	// Invert mouse coordinates
	mouseY = glView->height()-mouseY;

	// Get 3D mouse coordinates to construct a ray

	CVector<3,GLdouble> mousePos3DA;
	CVector<3,GLdouble> mousePos3DB;

	get3DMouseCoords(0.0,mousePos3DA.c[0],mousePos3DA.c[1],mousePos3DA.c[2]);
	get3DMouseCoords(1.0,mousePos3DB.c[0],mousePos3DB.c[1],mousePos3DB.c[2]);

	CVector<3,GLdouble> rayDirection = mousePos3DB-mousePos3DA;

	// initialize vectors describing a the controlgrid plane

	CVector<3,float> planeOrigin;
	CVector<3,float> planeU;
	CVector<3,float> planeV;

	planeOrigin.c[0]=0.0; planeOrigin.c[1]=0.0; planeOrigin.c[2]=0.0;
	planeU.c[0]=1.0; planeU.c[1]=0.0; planeU.c[2]=0.0;
	planeV.c[0]=0.0; planeV.c[1]=1.0; planeV.c[2]=0.0;

	// calculate the intersection plane coordinates
	// between the ray and the plane

	getPlaneRayIntersection(mousePos3DA,
															rayDirection,
															planeOrigin,
															planeU,
															planeV,
															ControlGridPlaneU,
															ControlGridPlaneV);
	// Invert mouse coordinates
	mouseY = glView->height()-mouseY;

};

void CBSplineWrapperEffect::getControlPointGridIndex()
{

	// Obtain the control point grid indices
	getControlPointAtMouse(PICK_THERSHOLD,
												 ControlGridPlaneU,
												 ControlGridPlaneV,
												 ControlPointCol,
												 ControlPointRow);
};

/*! Die Methode \b selectControlPoint() markiert den ber
 *  \b x und \b y spezifizierten Kontrollgitterpunkt.
 *  Dazu wird er in die Liste der markierten Punkte aufgenommen und
 *  seine Farbe auf die eines Markierten Punkts gesetzt.
 */
void CBSplineWrapperEffect::selectControlPoint(int x, int y)
{
	if ((x>=0) && (x<ControlMesh->width()) && (y>=0) && (y<ControlMesh->height()))
	{
		// Insert point to selection list
		SelectedControlPoints.insert(&ControlMesh->get(x,y));

		// Set control point color
		ControlPointColorGrid->set(x,y,SelectedControlMeshVertexColor);
	};
};

/*! Die Methode \b moveSelectedControlPoints() verschiebt alle markierten
 *  Kontrollgitterpunkte um den Vektor \b Delta.
 *  Dazu wird die Liste der markierten Punkte durchlaufen und auf jeden Punkt
 *  der Vektor \b Delta addiert.
 */
void CBSplineWrapperEffect::moveSelectedControlPoints(CVector<2,GLfloat> Delta)
{
	// Add the Delta vector to all selected control points
	for (int i = 0 ; i<SelectedControlPoints.count() ; i++)
	{
		*SelectedControlPoints[i]+=Delta;
	};
}

/*! Die Methode \b rotateSelectedControlPoints() rotiert alle markierten
 *  Kontrollgitterpunkte um den Punkt \b Center um den Winkel \b Angle.
 *  Dazu wird von jedem selektierte Punkt zunchst der Vektor \b Center
 *  abgezogen. Anschlieend wird jeder selektierte Punkt mit einer 2x2 Drehmatrix multipliziert.
 *  Schlielich wird der Vektor \b Center wieder auf jeden Punkt aufaddiert.
 */
void CBSplineWrapperEffect::rotateSelectedControlPoints(float Angle, CVector<2,GLfloat> Center)
{

	CVector<2,GLfloat> zeroVector;

	// Subtract center from point coordinates
	moveSelectedControlPoints(zeroVector-Center);

	// Create a rotation matrix
	CMatrix<2,GLfloat> rotMatrix;
	float cs = cos(Angle*3.141592654f/180.0f);
	float sn = sin(Angle*3.141592654f/180.0f);

	rotMatrix.c[0][0] = cs;
	rotMatrix.c[0][1] = -sn;
	rotMatrix.c[1][0] = sn;
	rotMatrix.c[1][1] = cs;

	// Apply matrix to all vertices
	for (int i = 0 ; i<SelectedControlPoints.count() ; i++)
	{
		*SelectedControlPoints[i]=rotMatrix*(*SelectedControlPoints[i]);
	};

	// Add center to point coordinates
	moveSelectedControlPoints(Center);

};

/*! Die Methode \b scaleSelectedControlPoints() skaliert alle markierten
 *  Kontrollgitterpunkte um den Punkt \b Center mit den Faktoren \b stretchX und \b stretchY.
 *  Dazu wird von jedem selektierte Punkt zunchst der Vektor \b Center
 *  abgezogen. Anschlieend wird jeder selektierte Punkt mit einer 2x2 Skalierungsmatrix multipliziert.
 *  Schlielich wird der Vektor \b Center wieder auf jeden Punkt aufaddiert.
 */
void CBSplineWrapperEffect::scaleSelectedControlPoints(float stretchX, float stretchY, CVector<2,GLfloat> Center)
{

	CVector<2,GLfloat> zeroVector;

	// Subtract center from point coordinates
	moveSelectedControlPoints(zeroVector-Center);

	// Create scale matrix
	CMatrix<2,float> scaleMatrix;
	scaleMatrix.c[0][0]=stretchX;
	scaleMatrix.c[1][1]=stretchY;

	// Apply matrix to all vertices
	for (int i = 0 ; i<SelectedControlPoints.count() ; i++)
	{
		*SelectedControlPoints[i]=scaleMatrix*(*SelectedControlPoints[i]);
	};

	// Add center to point coordinates
	moveSelectedControlPoints(Center);

};

/*! Die Methode \b resolveGridBindigs() lst alle Bindungen
 *  der verschiedenen Gitter auf. Hierzu werden die Bindungen
 *  an die \b CMeshPainter -Objekte aufgelst und ebenfalls
 *  an die Deformer-Klassen.
 */
void CBSplineWrapperEffect::resolveGridBindings()
{

	// Resolve mesh painter bindings

	ImageMeshPainter.setVertexGrid(NULL);
	ImageMeshPainter.setTexelGrid(NULL);

	ControlMeshPainter.setVertexGrid(NULL);
	ControlMeshPainter.setColorGrid(NULL);

	// Resolve deformer bindings

	for (int i = 0 ; i<Deformers.count() ; i++)
	{
		Deformers[i]->setSourceGrid(NULL);
		Deformers[i]->setControlGrid(NULL);
		Deformers[i]->setDestinationGrid(NULL);
	};

};

/*! Die Methode \b bindActiveDeformer() verbindet die Gitter
 *  mit dem gerade aktiven Deformationsalgorithmus, d.h.
 *  Das Bildgitter und das Kontrollgitter werden dem
 *  Algorithmus zugewiesen.
 *
 */
void CBSplineWrapperEffect::bindActiveDeformer()
{

	if (ActiveDeformer!=NULL)
	{

		// Bind control mesh and the image mesh to the active
		// deformer

		ActiveDeformer->setSourceGrid(ImageMesh);
		ActiveDeformer->setControlGrid(ControlMesh);
		ActiveDeformer->setDestinationGrid(ImageMesh);

		// The source mesh and the destination mesh may be
		// the same, since all deformers used create a
		// temporary copy of the source mesh to perform
		// their deformation

	};

};

/*! Die Methode \b unbindActiveDeformer() lst die zum
 *  aktiven Deformationsalgorithmus bestehenden Gitterverbindungen
 *  auf, d.h. das Kontrollgitter und das Bildgitter werden
 *  vom Algorithmus getrennt.
 *
 */
void CBSplineWrapperEffect::unbindActiveDeformer()
{

	if (ActiveDeformer!=NULL)
	{

		// Reset deformer bindings

		ActiveDeformer->setSourceGrid(NULL);
		ActiveDeformer->setControlGrid(NULL);
		ActiveDeformer->setDestinationGrid(NULL);

	};

};

/*! Die Methode \b enableControlMeshResolutionPanel() aktiviert oder
 *  deaktiviert die Spinboxen, welche fr die Einstellung der
 *  horizontalen und der vertikalen Kontrollgitterauflsung zustndig sind.
 */
void CBSplineWrapperEffect::enableControlMeshResolutionPanel(bool Enabled)
{

	spnHResCMesh->setEnabled(Enabled);
	spnVResCMesh->setEnabled(Enabled);

};

/*! Die Methode resizeControlMesh() ndert die Gre des Kontrollgitters.
 *
 *  Hierzu wird das Bildgitter zunchst zurckgesetzt, so dass es wieder
 *  regulr ist. Dann wird ein Backup des Kontrollgitters erzeugt.
 *  Es wird ebenfalls ein regulres temporres Kontrollgitter erzeugt,
 *  welches dieselben Ausmae aufweist wie das gerade aktive.
 *  Danach wird ein neues Kontrollgitter mit der neuen Auflsung erzeugt.
 *  Das neue Kontrolgitter wird nun an den aktiven Deformationsalgoritmus zusammen
 *  mit dem zurckgesetzen Bildgitter verbunden.
 *  Nun wird ein \b CLinearDeformer -Objekt dazu verwendet das neue Kontrollgitter
 *  durch mit dem alten zu verformen. Danach hat das neue Kontrollgitter eine hnliche Form, wie das alte.
 *  Die temporren Objekte werden freigegeben. Die neuen Gitter werden neu
 *  gebunden an die \b CGridPainter -Objekte.
 *
 *  Im Morphingmodus werden zustzlich die Texelkoordinaten neu berechnet.
 *
 *  Im B-Splinemodus werden neue Knotenmatrizen erzeugt und die Tabelle aktualisiert.
 *
 */
void CBSplineWrapperEffect::resizeControlMesh(int newWidth, int newHeight)
{

	if ((newWidth!=ControlMesh->width()) || (newHeight!=ControlMesh->height()))
	{

		// So this one is a little bit tricky!
		// First we make the image mesh a regular mesh again.
		// Then we create a backup
		// of the current control mesh state. We also create a regular
		// mesh with the current control mesh's dimensions
		// Then we create the new control mesh as a regular mesh.
		// Now we use the linear deformer object to deform the new
		// control mesh by the old control mesh. The result is
		// a control mesh with the new given dimension but a shape
		// similar to the old one

		// Unbind image mesh and the active deformer
		unbindActiveDeformer();
		ImageMeshPainter.setVertexGrid(NULL);

		// Delete image mesh and create new regular one
		int imgWidth = ImageMesh->width();
		int imgHeight = ImageMesh->height();
		delete ImageMesh;
		buildRegularMesh(ImageMesh,-1.0,-1.0,2.0,2.0,imgWidth,imgHeight);

		// Save Control mesh
		CGrid< CVector<2, GLfloat> >* ControlMeshBackup;
		ControlMeshBackup = ControlMesh;

		// Create a new control mesh
		buildRegularMesh(ControlMesh,-1.0,-1.0,2.0,2.0,newWidth,newHeight);

		// Rebind active deformer
		bindActiveDeformer();

		// Deform regular mesh and recalculate texel coordinates
		ActiveDeformer->deform();
		recalculateTexelCoordinates(*ImageMesh,*TexelMesh);

		// Create a temporary control mesh
		CGrid< CVector<2, GLfloat> >* TemporaryControlMesh;
		buildRegularMesh(TemporaryControlMesh,-1.0,-1.0,2.0,2.0,ControlMeshBackup->width(),ControlMeshBackup->height());

		// Create a linear deformer object to deform the current
		// control mesh by the old control mesh using the temporary control mesh

		CLinearDeformer<GLfloat> tempLinearDeformer;
		tempLinearDeformer.setSourceGrid(ControlMesh);
		tempLinearDeformer.setDestinationGrid(ControlMesh);
		tempLinearDeformer.setControlGrid(TemporaryControlMesh);
		// Set the control points of the temporary control mesh to the
		// points of the old control mesh
		*TemporaryControlMesh = *ControlMeshBackup;

		// Deform the new control mesh
		tempLinearDeformer.deform();

		// Resize control vertex color grid
		SelectedControlPoints.clear();
		ControlPointColorGrid->resize(ControlMesh->width(),ControlMesh->height());
		ControlPointColorGrid->clear(ControlMeshVertexColor);
		SelectedControlPoints.clear();

		// Destroy old and temporary control mesh
		delete ControlMeshBackup;
		delete TemporaryControlMesh;

		// Deform image mesh by the new control mesh
		ActiveDeformer->deform();

		// check if moprhing mode is active
		if (EffectMode == emMorphing)
		{
			// Recalculate texel coordinates
			recalculateTexelCoordinates(*ImageMesh,*TexelMesh);
		};

		// Rebind control mesh
		ControlMeshPainter.setVertexGrid(NULL);
		ControlMeshPainter.setVertexGrid(ControlMesh,2,GL_FLOAT);

		// Rebind image mesh
		ImageMeshPainter.setVertexGrid(ImageMesh,2,GL_FLOAT);

		// Update knot matrix table
		if (DeformationMode == dmBSpline)
		{
			knotMatrixToTable(ActiveKnotMatrix);
		};

		// Update spin buttons
		spnHResCMesh->setValue(ControlMesh->width());
		spnVResCMesh->setValue(ControlMesh->height());

	};

};

/*! Die Methode resizeImageMesh() ndert die Auflsung des Bildgitters.
 *  Hierzu wird zunchst der aktive Deformationsalgorithmus von
 *  allen gittern getrennt.
 *  Es wird ein Backup des Kontrollgitters erzeugt, danach wird
 *  das Kontrollgitter auf ein regulres Gitter zurckgesetzt.
 *  Nun wird ein neues Bildgitter mit der gegebenen Auflsung erzeugt.
 *  Ebenfalls werden die Texelgitter neu skaliert und mit Standardwerten belegt.
 *  Die Gitter werden wieder an den aktiven Deformer gebunden.
 *  Danach wird der alte Zustand der Kontrollgitterpunkte wiederhergestellt.
 *  Das Bildgitter wird dann mit dem aktiven Deformer neu berechnet.
 *  Zum Schluss werden die Bindungen an die \b CGridPainter -Objekte wiederhergestellt.
 *
 *  Im Morphingmodus werden zustzlich die Texelkoordinaten aller Texelgitter
 *  nach dem defomrieren neu berechnet, so da dei Textur unverzerrt erscheint.
 *
 */
void CBSplineWrapperEffect::resizeImageMesh(int newWidth, int newHeight)
{

	if ((ImageMesh->width()!=newWidth) || (ImageMesh->height()!=newHeight))
	{

		// Unbind active deformer
		unbindActiveDeformer();

		// Destroy current image mesh
		delete ImageMesh;

		// Create a new image mesh
		buildRegularMesh(ImageMesh,-1.0,-1.0,2.0,2.0,newWidth,newHeight);

		// Destroy current texel mesh
		delete TexelMesh;

		// Create a new texel mesh
		buildRegularMesh(TexelMesh,0.0,0.0,1.0,1.0,newWidth,newHeight);

		// Create a copy of the current control mesh
		CGrid< CVector<2, GLfloat> > ControlMeshBackup = *ControlMesh;

		// Build a temporary regular control mesh
		CGrid< CVector<2, GLfloat> >* TemporaryControlMesh;
		buildRegularMesh(TemporaryControlMesh,-1.0,-1.0,2.0,2.0,ControlMesh->width(),ControlMesh->height());

		// Assign regular mesh to control mesh
		*ControlMesh = *TemporaryControlMesh;

		// Rebind active deformer
		bindActiveDeformer();

		// Delete temporary control mesh
		delete TemporaryControlMesh;

		// check if moprhing mode is active
		if (EffectMode == emMorphing)
		{

			// Recalculate texel coordinates
			recalculateTexelCoordinates(*ImageMesh,*TexelMesh);

			// Resize source and destination texel mesh
			TexelMeshA->resize(ImageMesh->width(),ImageMesh->height());
			makeRegularMesh(TexelMeshA,0.0,0.0,1.0,1.0);
			TexelMeshB->resize(ImageMesh->width(),ImageMesh->height());
			makeRegularMesh(TexelMeshB,0.0,0.0,1.0,1.0);

			if (ActiveDeformer!=NULL)
			{

				// Deform by source control mesh
				*ControlMesh=( (CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.first() )->VertexGrid();
				ActiveDeformer->deform();
				recalculateTexelCoordinates(*ImageMesh,*TexelMeshA);

				// Deform by destination control mesh
				*ControlMesh=( (CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.last() )->VertexGrid();
				ActiveDeformer->deform();
				recalculateTexelCoordinates(*ImageMesh,*TexelMeshB);

			}

		}
		else
		{

			// Deform and recalculate texel mesh
			if (ActiveDeformer!=NULL) ActiveDeformer->deform();
			recalculateTexelCoordinates(*ImageMesh,*TexelMesh);

		};

		// Restore control mesh backup
		*ControlMesh = ControlMeshBackup;

		// Deform image mesh
		if (ActiveDeformer!=NULL) ActiveDeformer->deform();

		// Rebind mesh painter
		ImageMeshPainter.setVertexGrid(NULL);
		ImageMeshPainter.setTexelGrid(NULL);
		ImageMeshPainter.setVertexGrid(ImageMesh,2,GL_FLOAT);
		ImageMeshPainter.setTexelGrid(TexelMesh,2,GL_FLOAT);

	};

};

/*! Die Methode \b setActiveDeformer() ndert den aktiven Deformationsalgorithmus.
 *  Dabei werden zunchst die Gitterbindungen zum alten Algorithmus aufgelst.
 *  Danach wird ein Backup des Kontrollgitters erzeugt.
 *  Das Kontrollgitter und das Bildgitter werden anschlieend auf regulre
 *  Gitter zurckgesetzt.
 *  Nun wird der neue aktive Deformer gesetzt und die Gitter mit ihm verbunden.
 *  Danach wird der alte Zustand des Kontrollgitters wiederhergestellt und
 *  das Bildgitter mit dem neuen Deformationsalgorithmus verzerrt.
 *  Schlielich werden die Bindungen an die \b CGridPainter -Objekte wiederhergestellt.
 *
 *  Im Morphing-Modus werden die Texelkoordinaten aller Texelgitter nach dem Deformieren
 *  neu berechnet.
 *
 */
void CBSplineWrapperEffect::setActiveDeformer(CControlGridDeformer<2,GLfloat>* newDeformer)
{

	if (newDeformer!=ActiveDeformer)
	{

		// This one is again a little bit tricky.
		// First we unbind the active deformer
		// Then we create a backup of the current control mesh
		// Afterwards we create a new regular image mesh and
		// a new regular control mesh.
		// Now we set the activeDeformer to the given one and
		// rebind it. Then we restore the control mesh points
		// and deform the image mesh again

		// unbind active deformer
		unbindActiveDeformer();

		// unbind mesh painter;
		ImageMeshPainter.setVertexGrid(NULL);
		ControlMeshPainter.setVertexGrid(NULL);

		// Delete image mesh and create new regular one
		int imgWidth = ImageMesh->width();
		int imgHeight = ImageMesh->height();
		delete ImageMesh;
		buildRegularMesh(ImageMesh,-1.0,-1.0,2.0,2.0,imgWidth,imgHeight);

		// Backup current control mesh
		CGrid< CVector<2, GLfloat> >* ControlMeshBackup = ControlMesh;

		// Create a new regular control mesh
		buildRegularMesh(ControlMesh,-1.0,-1.0,2.0,2.0,ControlMeshBackup->width(),ControlMeshBackup->height());

		// set the active deformer to the given deformer
		ActiveDeformer = newDeformer;

		// rebind deformer
		bindActiveDeformer();

		// Check if morphing effect is active
		if (EffectMode == emMorphing)
		{
			recalculateMorphingTexelCoordinates();
		}

		// deform the image mesh and recalculate texel mesh
		ActiveDeformer->deform();
		recalculateTexelCoordinates(*ImageMesh,*TexelMesh);

		// restore the control mesh state
		*ControlMesh = *ControlMeshBackup;
		delete ControlMeshBackup;

		// deform the image mesh
		ActiveDeformer->deform();

		// rebind mesh painter
		ImageMeshPainter.setVertexGrid(ImageMesh,2,GL_FLOAT);
		ControlMeshPainter.setVertexGrid(ControlMesh,2,GL_FLOAT);

	};

};

/*! Die Methode \b lockBorderVertices() setzt die
 *  Randkontrollgitterpunkte an ihre ursprnglichen
 *  Koordinaten, so dass diese nur noch entlang der rechteckigen
 *  Gitterkanten verschoben werden knnen.
 *  Die Eckpunkte werden dadurch unbeweglich gemacht.
 */
void CBSplineWrapperEffect::lockBorderVertices()
{

	// Lock top and bottom border
	for (int x = 0 ; x<ControlMesh->width() ; x++)
	{

		CVector<2,GLfloat> vtx = ControlMesh->get(x,0);
		vtx.c[1] = -1.0f;
		// Clamp x-coordinate
		if (vtx.c[0]<-1.0f) vtx.c[0]=-1.0;
		if (vtx.c[0]>1.0f) vtx.c[0]=1.0;
		ControlMesh->set(x,0,vtx);

		vtx = ControlMesh->get(x,ControlMesh->height()-1);
		vtx.c[1] =  1.0f;
		// Clamp x-coordinate
		if (vtx.c[0]<-1.0f) vtx.c[0]=-1.0;
		if (vtx.c[0]>1.0f) vtx.c[0]=1.0;
		ControlMesh->set(x,ControlMesh->height()-1,vtx);

	};

	// Lock left and right border
	for (int y = 0 ; y<ControlMesh->height() ; y++)
	{

		CVector<2,GLfloat> vtx = ControlMesh->get(0,y);
		vtx.c[0] = -1.0f;
		// Clamp y-coordinate
		if (vtx.c[1]<-1.0f) vtx.c[1]=-1.0;
		if (vtx.c[1]>1.0f) vtx.c[1]=1.0;
		ControlMesh->set(0,y,vtx);

		vtx = ControlMesh->get(ControlMesh->width()-1,y);
		vtx.c[0] =  1.0f;
		// Clamp y-coordinate
		if (vtx.c[1]<-1.0f) vtx.c[1]=-1.0;
		if (vtx.c[1]>1.0f) vtx.c[1]=1.0;
		ControlMesh->set(ControlMesh->width()-1,y,vtx);

	};

};

/*! Die Methode \b intializeKeyFrameMode() initialisiert den
 *  Keyframe-Modus des Effekts. Dabei werden alle Keyframes
 *  gelscht, das Verschieben der Keyframes aktiviert und
 *  das Sperren der Randpunkte wiede deaktiviert.
 *  Der Framecursor wird an die Position 0 zurckgesetzt.
 *  Ebenfalls werden die Steuerelemente zum Einstellen
 *  der Kontrollgitterauflsung wieder aktiviert.
 */
void CBSplineWrapperEffect::initializeKeyFrameMode()
{

	// Clear all key frames
	KeyFrameList.clear();

	// Deactivate border locking and enable widget
	cbxLockBorder->setChecked(false);
	cbxLockBorder->setEnabled(true);

	// Enable keyframe movment
	usrKeyFrameBar->setFrameMoving(true);
	// Update keyframe bar
	usrKeyFrameBar->update();

	// Setup key frame bar view
	usrKeyFrameBar->setDisplayRange(-2.0,2.0);
	usrKeyFrameBar->setCursorPosition(0);
  usrKeyFrameBar->centerView();

	// rebuild texel mesh
	makeRegularMesh(TexelMesh,0.0,0.0,1.0,1.0);

	// Enable control mesh resolution panel
	enableControlMeshResolutionPanel(true);

};

/*! Die Methode \b intializeMorphingMode() initialisiert den
 *  Morphing-Modus des Effekts. Dabei werden zwei feste
 *  Keyframes an den Stellen 0.0 und 1.0 mit dem
 *  aktuellen Kontrollgitterzustand eingefgt.
 *  Das Verschieben der Keyframes wird deaktiviert,
 *  das Sperren der Randpunkte wiede aktiviert,
 *  Der Framecursor wird an die Position 0 zurckgesetzt.
 *  Ebenfalls werden die Steuerelemente zum Einstellen
 *  der Kontrollgitterauflsung wieder deaktiviert.
 *  Ferner werden die Texelkoordinaten neu berechnet, so dass
 *  die Verzerrung des Bildgitters keinerlei Auswirkungen zeigt.
 */
void CBSplineWrapperEffect::initializeMorphingMode()
{

	// Clear all key frames
	KeyFrameList.clear();

	// Insert current control mesh as source and destination frame
	KeyFrameList.insert(
		new CVertexGridKeyFrame<2,GLfloat>(0.0,
			new CGrid< CVector<2, GLfloat> >(*ControlMesh)));
	usrKeyFrameBar->update();

	KeyFrameList.insert(
		new CVertexGridKeyFrame<2,GLfloat>(1.0,
			new CGrid< CVector<2, GLfloat> >(*ControlMesh)));

	// Activate border locking and disable widget
	cbxLockBorder->setChecked(true);
	cbxLockBorder->setEnabled(false);

	// Disable keyframe movment
	usrKeyFrameBar->setFrameMoving(false);

	// Setup key framebar view
	usrKeyFrameBar->setDisplayRange(-0.2f,1.2f);
	usrKeyFrameBar->setCursorPosition(0);

	// Disable control grid resolution panel
	enableControlMeshResolutionPanel(false);

	// Recalculate texel coordinates
	recalculateTexelCoordinates(*ImageMesh,*TexelMesh);

	// Set texel meshes
	*TexelMeshA = *TexelMesh;
	*TexelMeshB = *TexelMesh;

	// Reset morphing time index
	MorphingTimeIndex = 0;

	// Fetch first image from source list
	texImage1 = parentFrame->fetchImage(0,0);

	// Set morphing edit state
	MorphingEditState = esSource;

};

/*! Die Methode \b recalculateTexelCoordinaes berechnet die
 *  Texelkoordinaten des Gitters \b TexelMesh anhand des Bildgitters
 *  \b ImageMesh neu. Dabei werden die Koordinaten einfach
 *  planar auf das \b ImageMesh neu projiziert, so dass anschlieend
 *  das sich darauf befindende Bild unverzerrt erscheint.
 */
void CBSplineWrapperEffect::recalculateTexelCoordinates(CGrid< CVector<2,GLfloat> >& ImageMesh, CGrid< CVector<2,GLfloat> >& TexelMesh)
{

	// Get image grid width, height and origin
	float width = ImageMesh.get(ImageMesh.width()-1,0).c[0]-ImageMesh.get(0,0).c[0];
	float height = ImageMesh.get(0,ImageMesh.height()-1).c[1]-ImageMesh.get(0,0).c[1];
	CVector<2,GLfloat> orig = ImageMesh.get(0,0);

	// Recalculate texel coordinates
	for (int y = 0 ; y<TexelMesh.height(); y++)
	{

		for (int x = 0 ; x<TexelMesh.width() ; x++)
		{

			CVector<2,GLfloat> res = ImageMesh.get(x,y)-orig;
			res.c[0] /= width;
			res.c[1] /= height;
			TexelMesh.set(x,y,res);

		};

	};

};

/*! Die Methode \b knotMatrixToTable() kopiert den Inhalt der durch \b matrixType
 *  spezifizierten KnotenMatrix des B-Spline-Alogrithmus in das Tabellen-Widget
 *  des Effekttoolfensters.
 */
void CBSplineWrapperEffect::knotMatrixToTable(enmKnotMatrixType matrixType)
{

	if (DeformationMode == dmBSpline)
	{

		// Get matrix

		CGrid<float>* matrix;

		switch (matrixType)
		{
			case kmCol:
				matrix = &((CBSplineDeformer<GLfloat>*) ActiveDeformer)->colKnotMatrix();
			break;

			case kmRow:
				matrix = &((CBSplineDeformer<GLfloat>*) ActiveDeformer)->rowKnotMatrix();
			break;

		};

		// Resize table
		tabKnotMatrix->setNumRows(matrix->height());
		tabKnotMatrix->setNumCols(matrix->width());

		int matrixHeight = matrix->height()-1;

		// Resize cells
		for (int x=0; x<tabKnotMatrix->numCols() ; x++)
		{
			tabKnotMatrix->setColumnWidth(x,64);
		};

		// Fill cells with matrix values
		for (int y=0; y<matrix->height(); y++)
		{
			for (int x=0; x<matrix->width(); x++)
			{
				tabKnotMatrix->setText(y,x,QString::number(matrix->get(x,matrixHeight-y)));
			}
		};

	};

};

/*! Die Methode \b recalculateMorphingTexelCoordinates() berechnet
 *  die Quell- und Zeilmorphingtexelkoordinaten neu.
 *  Zuerst wird dazu ein Backup des Kontrollgitters erzeugt.
 *  Danach wird dem Kontrollgitter das Quellmorphinggitter
 *  zugewiesen. Das Bildgitter wird nun mit dem aktiven
 *  Deformer verzerrt. Jetzt werden die Quellmorphingtexelkoordinaten
 *  neu berechnet.
 *  Anschlieend wird dem Kontrollgitter das Zielmorphinggitter
 *  zugewiesen und dieselbe Prozedur fr die Zielmorphingtexelkoordinaten
 *  wiederholt.
 *  Zuletzt wird der alte Zustand des Kontrollgitters wiederhergestellt.
 *
 */
void CBSplineWrapperEffect::recalculateMorphingTexelCoordinates()
{

	// Backup control mesh
	CGrid< CVector<2,GLfloat> > ControlMeshBackup = *ControlMesh;

	// Recalculate source and destination texel mesh
	*ControlMesh=((CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.first())->VertexGrid();
	ActiveDeformer->deform();
	recalculateTexelCoordinates(*ImageMesh,*TexelMeshA);

	*ControlMesh=((CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.last())->VertexGrid();
	ActiveDeformer->deform();
	recalculateTexelCoordinates(*ImageMesh,*TexelMeshB);

	// Restore control mesh
	*ControlMesh = ControlMeshBackup;

};

/*! Die Methode \b enableKnotMatrixPanel aktiviert oder
 *  oder deaktiviert die Steuerelemente die zur Bearbeitung
 *  der Knotenmatrizen dienen.
 */
void CBSplineWrapperEffect::enableKnotMatrixPanel(bool Enabled)
{

	// Enable/Disable knot matrix widgets
	grpNodeMatrix->setEnabled(Enabled);
	tabKnotMatrix->setEnabled(Enabled);
	btnResetKnotMatrix->setEnabled(Enabled);
	btnMatrixType->setEnabled(Enabled);

	if (!Enabled)
	{
		tabKnotMatrix->setNumCols(0);
		tabKnotMatrix->setNumRows(0);
	}
	else
	{
		knotMatrixToTable(ActiveKnotMatrix);
	};

};

/*! Der Event-Handler \b mousePressEvent() behandelt
 *  das Niederdrcken von Maustasten.
 *  Hier wird das selektieren von Punkten implementiert.
 *  Ebenfalls werden hier die Kontrollpunktmanipulationsmodi
 *  initialisiert.
 *
 *  \remarks
 *  Die Mouseevents werden ignoriert, wenn gerade
 *  eine Animation abgespielt wird.
 */

void CBSplineWrapperEffect::mousePressEvent(QMouseEvent* event)
{

	// Check if animation is running
	if (!tmrEffectTimer->isActive())
	{

		// Call base class implementation of this function
		GLEffect::mouseMoveEvent(event);

		// Project mouse coordinates onto the control grid plane
		getControlPlaneCoordinates();

		// Obtain control point under the mouse cursor
		getControlPointGridIndex();

		if (event->button() == LeftButton)
		{

			// *****************************************************************
			// *** Here the control point selection is beeing handled        ***
			// *****************************************************************

			// Check if shift button is pressed
			if (!(event->state() & ShiftButton))
			{
					// Unselect all points
					unselectAll();
			};

			// Check if a control point was hit
			if ((ControlPointCol!=-1) && (ControlPointRow!=-1))
			{

				// Check if vertex is already selected
				if (ControlPointColorGrid->get(ControlPointCol,ControlPointRow)!=SelectedControlMeshVertexColor)
				{

					// Select control vertex
					selectControlPoint(ControlPointCol,ControlPointRow);

				};

			};

			// ***************************************************************
			// *** Here the setup of any control point translation is      ***
			// *** beeing handled.                                         ***
			// ***************************************************************

			if (SelectedControlPoints.count()>0)
			{

				switch (TranslationMode)
				{

					// Control point movement
					case tmMove:
					{

						// Setup backup plane coordinates
						OldControlGridPlaneU = ControlGridPlaneU;
						OldControlGridPlaneV = ControlGridPlaneV;

						// Set translation state
						TranslationState = tsMoving;

						// Grab mouse
						glView->grabMouse();

					}
					break;

					// Control point rotation
					case tmRotate:
					{

						// Setup backup plane coordinates
						OldControlGridPlaneU = ControlGridPlaneU;
						OldControlGridPlaneV = ControlGridPlaneV;

						// Setup translation center coordinates
						TranslationCenterU = ControlGridPlaneU;
						TranslationCenterV = ControlGridPlaneV;

						// Set translation state
						TranslationState = tsRotating;

						// Grab mouse
						glView->grabMouse();

					}
					break;

					// Control point scaling
					case tmScale:
					{

						// Setup backup plane coordinates
						OldControlGridPlaneU = ControlGridPlaneU;
						OldControlGridPlaneV = ControlGridPlaneV;

						// Setup translation center coordinates
						TranslationCenterU = ControlGridPlaneU;
						TranslationCenterV = ControlGridPlaneV;

						// Set translation state
						TranslationState = tsScaling;

						// Grab mouse
						glView->grabMouse();

					}
					break;

				};

			};

		};

		// Handle marking box

		if (event->button() == RightButton)
		{

			// Setup marking box left corner
			LeftCornerU = RightCornerU = ControlGridPlaneU;
			LeftCornerV = RightCornerV = ControlGridPlaneV;

			// Set translation state
			TranslationState = tsBoxing;

			// Grab mouse
			glView->grabMouse();

		};

		// Update view
		update();

	};

};

/*! Der Event-Handler \b mouseMoveEvent() behandelt
 *  das verschieben des Mauszeigers.
 *  Hier wird das Transformieren von Kontrollpunkten und das
 *  ziehen einer Selektionsbox implementiert.
 *
 *  \remarks
 *  Die Mouseevents werden ignoriert, wenn gerade
 *  eine Animation abgespielt wird.
 */
void CBSplineWrapperEffect::mouseMoveEvent(QMouseEvent* event)
{

	// Check if animation is running
	if (!tmrEffectTimer->isActive())
	{

		// Call base class implementation of this function
		GLEffect::mouseMoveEvent(event);

		// Project mouse coordinates onto the control grid plane
		getControlPlaneCoordinates();

		// Obtain control point under the mouse cursor
		getControlPointGridIndex();

		// ****************************************************************
		// *** Here all the translation modes are beeing handled        ***
		// ****************************************************************

		if (TranslationState != tsNothing)
		{

			switch (TranslationState)
			{
				// Moving...
				case tsMoving:
				{

					// Calculate Delta vector
					CVector<2,GLfloat> deltaVector;
					deltaVector.c[0]= ControlGridPlaneU-OldControlGridPlaneU;
					deltaVector.c[1]= ControlGridPlaneV-OldControlGridPlaneV;

					// Store current coordinates as backup coordintes
					OldControlGridPlaneU = ControlGridPlaneU;
					OldControlGridPlaneV = ControlGridPlaneV;

					// Move selected points by the delta vector
					moveSelectedControlPoints(deltaVector);

					// Lock vertices if lock activated
					if (LockBorder) lockBorderVertices();

				}
				break;

				// Rotating...
				case tsRotating:
				{

					// Create center vector
					CVector<2, float> center;

					center.c[0]=TranslationCenterU;
					center.c[1]=TranslationCenterV;

					// Rotate selected points by the deltax value around
					// the center
					rotateSelectedControlPoints((ControlGridPlaneU-OldControlGridPlaneU)*50.0f,center);

					// Lock vertices if lock activated
					if (LockBorder) lockBorderVertices();

					// Store current coordinates as backup coordintes
					OldControlGridPlaneU = ControlGridPlaneU;
					OldControlGridPlaneV = ControlGridPlaneV;

				}
				break;

				// Scaling...
				case tsScaling:
				{

					// Create center vector
					CVector<2, float> center;

					center.c[0]=TranslationCenterU;
					center.c[1]=TranslationCenterV;

					// Scale selected points by the delta on
					// the center
					scaleSelectedControlPoints(
							ControlGridPlaneU-OldControlGridPlaneU+1.0f,
							ControlGridPlaneV-OldControlGridPlaneV+1.0f,center);

					// Lock vertices if lock activated
					if (LockBorder) lockBorderVertices();

					// Store current coordinates as backup coordintes
					OldControlGridPlaneU = ControlGridPlaneU;
					OldControlGridPlaneV = ControlGridPlaneV;

				}
				break;

				// Boxing...
				case tsBoxing:
				{
					// Set right corner of marking box
					RightCornerU = ControlGridPlaneU;
					RightCornerV = ControlGridPlaneV;
				}
				break;

			};

		}
		else
		{
			// Change mouse cursor shape depending on if a control point
			// has been hit
			if ((ControlPointCol==-1) && (ControlPointRow==-1))
				glView->unsetCursor();
			else
				glView->setCursor(pointingHandCursor);
		};

	};

};

/*! Der Event-Handler \b mouseReleaseEvent() behandelt
 *  das Loslassen von Maustasten.
 *  Hier wird der Bearbeitungszustand zurckgesetzt und der
 *  GLView aktualisiert.
 *
 *  \remarks
 *  Die Mouseevents werden ignoriert, wenn gerade
 *  eine Animation abgespielt wird.
 */
void CBSplineWrapperEffect::mouseReleaseEvent(QMouseEvent* Event)
{

	// Check if animation is running
	if (!tmrEffectTimer->isActive())
	{

		// Call base class implementation of method
		GLEffect::mouseReleaseEvent(Event);

		// Check if any control point translation has been
		// performed

		if (TranslationState != tsNothing)
		{

			// Check if a mark box has been drawn then select marked vertices
			if (TranslationState == tsBoxing)
			{
				selectBoxedVertices(!(Event->state() & ShiftButton));
			};

			// Reset translation state
			TranslationState = tsNothing;

			// Release mouse
			glView->releaseMouse();

			// Update view
			update();

		};

	};

};

/*! Der Event-Handler \b mousePressEvent() behandelt
 *  das Drcken von Tasten.
 *  Hier werden die in der Anwendung zugnglichen Schnelltasten
 *  implementiert, welche die wichtigsten Optionen aktivieren.
 *
 */
void CBSplineWrapperEffect::keyPressEvent(QKeyEvent* event)
{

	// Check if any translation action is beeing performed
	if (TranslationState == tsNothing)
	{

		// Process key events, to activate or toggle
		// editing modes
		switch (event->key())
		{

			// Toggle image paint mode
			case Qt::Key_1:
				setPaintImage(!PaintImage);
			break;

			// Toggle image mesh paint mode
			case Qt::Key_2:
				setPaintImageMesh(!PaintImageMesh);
			break;

			// Toggle control mesh paint mode
			case Qt::Key_3:
				setPaintControlMesh(!PaintControlMesh);
			break;

			// Toggle control mesh paint mode
			case Qt::Key_4:
				setPaintControlMeshVertices(!PaintControlMeshVertices);
			break;

			// Move
			case Qt::Key_M:
				// Set translation mode to 'move'
				setTranslationMode(tmMove);
			break;

			// Scale
			case Qt::Key_S:
				// Set translation mode to 'scale'
				setTranslationMode(tmScale);
			break;

			// Rotate
			case Qt::Key_R:
				// Set translation mode to 'rotate'
				setTranslationMode(tmRotate);
			break;

			// Linear mode
			case Qt::Key_L:
				// Set deformation mode to 'linear'
				setDeformationMode(dmLinear);
			break;

			// B-Spline mode
			case Qt::Key_B:
				// Set deformation mode to 'B-Spline'
				setDeformationMode(dmBSpline);
			break;

			// Select all control vertices
			case Qt::Key_A:
				selectAll();
			break;

			// Unselect all control vertices
			case Qt::Key_U:
				unselectAll();
			break;

			// Insert key frame
			case Qt::Key_Space:
				if (EffectMode==emKeyFrame)
				{
					// Create a key frame
					insertKeyFrame();
				};
			break;

		};

	};

};

/*! Die Methode \b setAnimationSpeed() setzt die Abspielgeschwindigkeit
 *  fr Animationen.
 */
void CBSplineWrapperEffect::setAnimationSpeed(int Speed)
{
	AnimationSpeed = Speed;
}

/*! Die Methode \b setAnimationMode() setzt einen der drei Animationsmodi.
 *  \note
 *  Der Parameter \b Mode sollte ein Wert der Enumeration \b enmAnimationMode sein.
 */
void CBSplineWrapperEffect::setAnimationMode(int Mode)
{
	if (AnimationMode != (enmAnimationMode) Mode)
	{
		AnimationMode = (enmAnimationMode) Mode;

		// Set button caption
		switch (AnimationMode)
		{
			// Once
			case amOnce:
			{
				btnAnimationMode->setText("Once");
				// set animation direction
				AnimationDirection=+1.0;
			}
			break;

			// Loop
			case amLoop:
			{
				btnAnimationMode->setText("Loop");
				// set animation direction
				AnimationDirection=+1.0;
			}
			break;

			// Ping Pong
			case amPingPong:
			{
				btnAnimationMode->setText("Ping-Pong");
			}
			break;

		};

	};
};

/*! Die Methode \b animate() aktualisiert die Animation.
 *  ber die \b interpolate() Methode wird der neue Zustand
 *  berechnet. Danach wird die Framecursorposition aktualisiert
 *  in Abhngigkeit vom gesetzten Animationsmodus.
 */
void CBSplineWrapperEffect::animate()
{

	// interpolate control mesh
	interpolate(usrKeyFrameBar->cursorPosition());

	// move frame cursor
	usrKeyFrameBar->setCursorPosition(usrKeyFrameBar->cursorPosition()+(AnimationDirection*AnimationSpeed/1000.0f));

	// Check animation mode
	switch (AnimationMode)
	{
		// Once
		case amOnce:
		{
			if (usrKeyFrameBar->cursorPosition()>KeyFrameList.last()->timeIndex())
			{
					// Move cursor to last frame
					usrKeyFrameBar->setCursorPosition(KeyFrameList.last()->timeIndex());

					// Stop animation
					this->stop();

			};
		}
		break;

		// Loop
		case amLoop:
		{
			if (usrKeyFrameBar->cursorPosition()>KeyFrameList.last()->timeIndex())
			{
				// Move cursor to first frame
				usrKeyFrameBar->setCursorPosition(KeyFrameList.first()->timeIndex());
			};
		}
		break;

		// Ping Pong
		case amPingPong:
		{

			// Switch Animation direction
			if (usrKeyFrameBar->cursorPosition()>KeyFrameList.last()->timeIndex())
			{
				// Move cursor to last frame
				usrKeyFrameBar->setCursorPosition(KeyFrameList.last()->timeIndex());
				AnimationDirection=-AnimationDirection;
			};

			if (usrKeyFrameBar->cursorPosition()<KeyFrameList.first()->timeIndex())
			{
				// Move cursor to first frame
				usrKeyFrameBar->setCursorPosition(KeyFrameList.first()->timeIndex());
				AnimationDirection=-AnimationDirection;
			};

		}
		break;

	};

};

/*! Die Methode \b finishInterpolate() finalisiert das
 *  manuelle Interpolieren. Im Morphing-Modus wird
 *  dabei zu dem Kontrollgitter zurckgesprungen,
 *  welches zuletzt bearbeitet wurde, d.h. entweder
 *  zum Quellkontrollgitter, oder zum Zielkontrollgitter.
 */
void CBSplineWrapperEffect::finishInterpolate()
{
	// Check effect mode
	switch (EffectMode)
	{

		// Morphing mode
		case emMorphing:
		{

			// restore editing state
			switch (MorphingEditState)
			{
				// restore source state
				case esSource:
				{
					sourceToControlMesh();
				}
				break;

				// restore destination state
				case esDestination:
				{
					destinationToControlMesh();
				}
				break;

			};

		}
		break;

	};

};

/*! Die Methode \b interpolate() interpoliert das Kontrollgitter
 *  am Zeitpunkt \b TimeIndex.
 *  Dabei werden zunchst das Keyframepaar ermittelt, in dessen
 *  Zeitintervall sich \b TimeIndex befindet.
 *  Danach wird mittels der Klasse \b CGridInterpolator das
 *  Kontrollgitter zwischen den beiden Keyframes interpoliert.
 *  Anschlieend wird das Bildgitter neu deformiert.
 *  Der Effekt wird danach gerendert.
 *  Die Methode sorgt auch dafr, dass im Falle von geladenen
 *  Bildsequenzen das zum Zeitpunkt \b Timeindex zugehrige
 *  Bild ermittelt wird. Die Bilder der Bildsequenzen werden
 *  dabei gleichmig auf den durch die Keyframes gebildete
 *  Gesamtzeitspanne verteilt.
 */
void CBSplineWrapperEffect::interpolate(float TimeIndex)
{

	// Check effect mode
	switch (EffectMode)
	{

		// Key frame mode
		case emKeyFrame:
		{
			// check first if there are any keyframes
			if (KeyFrameList.count()>0)
			{

				// Handle one frame only
				if (KeyFrameList.count()==1)
				{
					*ControlMesh=( (CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.first() )->VertexGrid();
					update();
				}
				else
				{
					// Find first frame with time index >= <TimeIndex>
					KeyFrameList.first();

					while((KeyFrameList.current()!=NULL) && (KeyFrameList.current()->timeIndex()<TimeIndex))
					{
						KeyFrameList.next();
					}

					// Get references to start and end frame
					CVertexGridKeyFrame<2,GLfloat>* DestinationFrame = (CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.current();
					CVertexGridKeyFrame<2,GLfloat>* SourceFrame = (CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.previous();

					if (SourceFrame == NULL)
					{

						if (DestinationFrame == NULL)
						{
							// Cursor lies before all frames
							*ControlMesh=( (CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.last() )->VertexGrid();
							update();

						}
						else
						{
							// Cursor lies past all frames
							*ControlMesh=( (CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.first() )->VertexGrid();
							update();

						}
					}
					else
					{

						// Create an interpolator object
						CGridInterpolator< CVector<2,GLfloat> > tempInterpolator;
						tempInterpolator.setSourceGrid(&SourceFrame->VertexGrid());
						tempInterpolator.setDestinationGrid(&DestinationFrame->VertexGrid());
						tempInterpolator.setResultGrid(ControlMesh);

						// Interpolate
						tempInterpolator.interpolate((TimeIndex-SourceFrame->timeIndex())/(DestinationFrame->timeIndex()-SourceFrame->timeIndex()));

					};

					// get texture frame index
					float fst = KeyFrameList.first()->timeIndex();
					float lst = KeyFrameList.last()->timeIndex();
					float cpos = usrKeyFrameBar->cursorPosition();
					float animLength = lst-fst;

					if (cpos<fst)
					{
						// fetch first frame
						texImage1 = parentFrame->fetchImage(0,0);
					}
					else
					if (cpos>lst)
					{
						// fetch last frame
						texImage1 = parentFrame->fetchImage(0,parentFrame->getNumImages(0)-1);
					}
					else
					{

						float imgindex;

						// Get image index
						if (animLength>0)
						{
							// Number of images * number of loops
							float nmimgs = (parentFrame->getNumImages(0))*spnFrameLoop->value();
							imgindex = ((cpos-fst)*nmimgs)/animLength;
							if (imgindex<0) imgindex = 0;
							if (imgindex>nmimgs-1) imgindex=nmimgs-1;
						}
						else
						{
							imgindex = 0;
						};

						// fetch image with the calculated index
						texImage1 = parentFrame->fetchImage(0,(int) imgindex % parentFrame->getNumImages(0) );

					};

					// Update view
					update();

				};

			}
			else
			{
				// Select first frame from list
				texImage1 = parentFrame->fetchImage(0,0);
				update();
			};

		};
		break;

		// Morphing mode
		case emMorphing:
		{

			// Clamp time index
			if (TimeIndex<0.0f) TimeIndex = 0.0f;
			if (TimeIndex>1.0f) TimeIndex = 1.0f;

			MorphingTimeIndex = TimeIndex;

			// Create an interpolator object
			CGridInterpolator< CVector<2,GLfloat> > tempInterpolator;
			tempInterpolator.setSourceGrid(&((CVertexGridKeyFrame<2,GLfloat>*)KeyFrameList.first())->VertexGrid());
			tempInterpolator.setDestinationGrid(&((CVertexGridKeyFrame<2,GLfloat>*)KeyFrameList.last())->VertexGrid());
			tempInterpolator.setResultGrid(ControlMesh);

			// Interpolate
			tempInterpolator.interpolate(TimeIndex);

			// Calculate image indices
			int indexA = (int) (TimeIndex*((float) (parentFrame->getNumImages(0)*spnFrameLoop->value())));
			int indexB = (int) (TimeIndex*((float) (parentFrame->getNumImages(1)*spnFrameLoop->value())));

			// Fetch images
			texImage1 = parentFrame->fetchImage(0,indexA % parentFrame->getNumImages(0));
			texImage2 = parentFrame->fetchImage(1,indexB % parentFrame->getNumImages(1));

			TranslationState = tsInterpolating;

			// Update view
			update();

		}
		break;

	};

};

/*! Die Methode \b setSplineOrder() ndert die Ordnung
 *  der B-Spline-Kurven.
 *  Im Falle, dass Keyframes gesetzt wurden wird
 *  der Wert \b Order auf das Minmum der Kontrollgittergre
 *  limitiert.
 *  Ebenfalls werden die Spinboxen zur Einstellung der
 *  Kontrolgitterauflsung nach unten hin durch \b Order beschrnkt.
 *  Die Routine setzt anschlieend die Ordnung des
 *  B-Spline-Deformationsalgorithmus und berechnet die Deformation neu.
 *  Ebenfalls wird die Ansicht der Knotenmatrixtabelle aktualisiert.
 *
 *  Falls Morphing aktiviert ist, so werden die Koordinaten aller
 *  Texelgitter neu berechnet.
 *
 */
void CBSplineWrapperEffect::setSplineOrder(int Order)
{

	if (Order!=CurveOrder)
	{

		// Check if any keyframes have been inserted
		if (KeyFrameList.count()>0)
		{

			// Check if curve-order exceeds control grid size limits
			if (Order>ControlMesh->height())
			{

				Order = ControlMesh->height();

			}

			if (Order>ControlMesh->width())
			{

				Order = ControlMesh->width();

			};

		};

		CurveOrder = Order;

		// Set order to the b-spline deformer
		if (DeformationMode == dmBSpline)
		{

			// Update spin button
			spnOrder->setValue(Order);

			// Adjust control grid size. The width and height must not
			// be smaller than the curve order

			if (ControlMesh->width()<CurveOrder)
			{
				setControlMeshHorizontalResolution(CurveOrder);
			};

			if (ControlMesh->height()<CurveOrder)
			{
				setControlMeshVerticalResolution(CurveOrder);
			};

			// Limit control mesh resolution spin buttons
			spnHResCMesh->setMinValue(CurveOrder);
			spnVResCMesh->setMinValue(CurveOrder);

			((CBSplineDeformer<GLfloat>*) ActiveDeformer)->setOrder(CurveOrder);

			// Backup control mesh and make it regular
			CGrid< CVector<2,GLfloat> > ControlMeshBackup = *ControlMesh;
			makeRegularMesh(ControlMesh,-1.0,-1.0,2.0,2.0);

			// Now deform the image mesh and recalculate texel coordinates
			ActiveDeformer->deform();
			recalculateTexelCoordinates(*ImageMesh,*TexelMesh);

			// Recaluclate morhping texel coordinates when in morphing mode
			if (EffectMode == emMorphing)
			{
				recalculateMorphingTexelCoordinates();
			};

			// Restore the control mesh and deform the image again
			*ControlMesh = ControlMeshBackup;
			ActiveDeformer->deform();

			// Update view
			update();

		};

		// Update knot matrix table
		knotMatrixToTable(ActiveKnotMatrix);

	};

};

/*! Die Methode \b selectBoxedVertices() ermittelt
 *  alle Kontrollpunkte, die sich in der Selektionsbox
 *  befinden und fgt diese zur Selektion hinzu.
 *  Der Parameter \b clear legt fest, ob zuvor die
 *  bereits bestehende Selektion zurckgesetzt werden soll.
 *
 */
void CBSplineWrapperEffect::selectBoxedVertices(bool clear)
{

	// Prepare box corners
	if (LeftCornerU>RightCornerU)
	{
		float tmp = LeftCornerU;
		LeftCornerU = RightCornerU;
		RightCornerU =  tmp;
	};

	if (LeftCornerV>RightCornerV)
	{
		float tmp = LeftCornerV;
		LeftCornerV = RightCornerV;
		RightCornerV =  tmp;
	};

	// Clear selection
	if (clear)
	{
		SelectedControlPoints.clear();
		ControlPointColorGrid->clear(ControlMeshVertexColor);
	};

	for (int y=0 ; y<ControlMesh->height(); y++)
	{

		for (int x=0 ; x<ControlMesh->width(); x++)
		{
			CVector<2,GLfloat> vtx = ControlMesh->get(x,y);

			// Check if point is inside the marking box
			if (
				  (vtx.c[0]>=LeftCornerU) &&
				  (vtx.c[0]<=RightCornerU) &&
					(vtx.c[1]>=LeftCornerV) &&
					(vtx.c[1]<=RightCornerV)
				 )
			{
				// Mark point
				ControlPointColorGrid->set(x,y,SelectedControlMeshVertexColor);
				SelectedControlPoints.insert(&ControlMesh->get(x,y));
			}

		};

	};

};

/*! Die Methode \b selectAll() fgt alle Kontrollpunkte
 *  zur Selektion hinzu.
 */
void CBSplineWrapperEffect::selectAll()
{

	ControlPointColorGrid->clear(SelectedControlMeshVertexColor);

	// Insert all control vertices into the selection list

	for (int y=0 ; y<ControlMesh->height(); y++)
	{
		for (int x=0 ; x<ControlMesh->width(); x++)
		{
			SelectedControlPoints.insert(&ControlMesh->get(x,y));
		}
	};

	update();

};

/*! Die Methode \b unselectAll() setzt die Selektion aller
 *  Kontrollpunkte zurck.
 */
void CBSplineWrapperEffect::unselectAll()
{

	// Check first if any points are selected

	if (SelectedControlPoints.count()!=0)
	{
		// Clear the control point color table
		ControlPointColorGrid->clear(ControlMeshVertexColor);
		// Clear the selection list
		SelectedControlPoints.clear();
	};

	update();

};

/*! Die Methode \b setBorderLock() aktiviert, oder deaktiviert
 *  die Bewegungsbeschrnkung fr die Kontrollgitterandpunkte.
 *  Beim aktivieren setzt diese gleichzeitig die Randpunkte an
 *  ihre ursprngliche Position zurck.
 */
void CBSplineWrapperEffect::setBorderLock(bool Enabled)
{
	if (Enabled!=LockBorder)
	{
		LockBorder=Enabled;
		cbxLockBorder->setChecked(Enabled);
		lockBorderVertices();
		update();
	};
};

/*! Die Methode \b setPaintImage() aktiviert, oder deaktiviert
 *  das Zeichnen des Bildes
 */
void CBSplineWrapperEffect::setPaintImage(bool Enabled)
{

	if (Enabled!=PaintImage)
	{
		PaintImage = Enabled;
		cbxPaintImage->setChecked(Enabled);
		update();
	};

};

/*! Die Methode \b setPaintImageMesh() aktiviert, oder deaktiviert
 *  das Zeichnen des Bildgitters.
 */
void CBSplineWrapperEffect::setPaintImageMesh(bool Enabled)
{

	if (Enabled!=PaintImageMesh)
	{
		PaintImageMesh = Enabled;
		cbxPaintImageMesh->setChecked(Enabled);
		update();
	};

};

/*! Die Methode \b setControlMesh() aktiviert, oder deaktiviert
 *  das Zeichnen des Kontrollgitters.
 */
void CBSplineWrapperEffect::setPaintControlMesh(bool Enabled)
{

	if (Enabled!=PaintControlMesh)
	{
		PaintControlMesh = Enabled;
		cbxPaintControlMesh->setChecked(Enabled);
		update();
	};

};

/*! Die Methode \b setControlMeshVertices() aktiviert, oder deaktiviert
 *  das Zeichnen der Kontrollpunkte.
 */
void CBSplineWrapperEffect::setPaintControlMeshVertices(bool Enabled)
{
	if (Enabled!=PaintControlMeshVertices)
	{
		PaintControlMeshVertices = Enabled;
		cbxPaintControlMeshVertices->setChecked(Enabled);
		update();
	};
};

/*! Die Methode \b setControlMeshHorizontalResolution() ndert die
 *  horizontale Auflsung des Kontrollgitters.
 *  Hierzu wird die Methode \b resizeControlMesh() aufgerufen.
 *  Der Effekt wird anschlieend neu gerendert.
 */
void CBSplineWrapperEffect::setControlMeshHorizontalResolution(int NewResolution)
{

	if (NewResolution!=	ControlMesh->width())
	{

		// Check if horizontal resolution is smaller than
		// the curve order
		if (DeformationMode == dmBSpline)
		{
			if (NewResolution<CurveOrder) NewResolution=CurveOrder;
		};

		// resize control mesh
		resizeControlMesh(NewResolution,ControlMesh->height());

		// update view
		update();

	};

};

/*! Die Methode \b setControlMeshVerticalResolution() ndert die
 *  vertikale Auflsung des Kontrollgitters.
 *  Hierzu wird die Methode \b resizeControlMesh() aufgerufen.
 *  Der Effekt wird anschlieend neu gerendert.
 */
void CBSplineWrapperEffect::setControlMeshVerticalResolution(int NewResolution)
{

	if (NewResolution!=	ControlMesh->height())
	{

		// Check if horizontal resolution is smaller than
		// the curve order
		if (DeformationMode == dmBSpline)
		{
			if (NewResolution<CurveOrder) NewResolution=CurveOrder;
		};

		// resize control mesh
		resizeControlMesh(ControlMesh->width(),NewResolution);

		// update view
		update();

	};

};

/*! Die Methode \b setImageMeshHorizontalResolution() ndert die
 *  horizontale Auflsung des Bildgitters.
 *  Hierzu wird die Methode \b resizeImageMesh() aufgerufen.
 *  Der Effekt wird anschlieend neu gerendert.
 */
void CBSplineWrapperEffect::setImageMeshHorizontalResolution(int NewResolution)
{

	if (NewResolution!=	ImageMesh->width())
	{

		// resize image mesh
		resizeImageMesh(NewResolution,ImageMesh->height());

		// update view
		update();
	};

};

/*! Die Methode \b setImageMeshVerticalResolution() ndert die
 *  vertikale Auflsung des Bildgitters.
 *  Hierzu wird die Methode \b resizeImageMesh() aufgerufen.
 *  Der Effekt wird anschlieend neu gerendert.
 */
void CBSplineWrapperEffect::setImageMeshVerticalResolution(int NewResolution)
{

	if (NewResolution!=	ImageMesh->height())
	{

		// resize image mesh
		resizeImageMesh(ImageMesh->width(),NewResolution);

		// update view
		update();

	};

};

/*! Die Methode \b setTranslationMode() ndert den
 *  Punktmanipulationsmodus und aktualisiert die zugehrige.
 *  Schaltflche.
 *
 *  \note
 *  Der Parameter \b Mode sollte ein Wert der Enumeration \b enmTranslationMode sein.
 */
void CBSplineWrapperEffect::setTranslationMode(int Mode)
{

	if (TranslationMode!=(enmTranslationMode) Mode)
	{

		// Set translation mode
		TranslationMode = (enmTranslationMode) Mode;

		// Set button caption in depenence of the selected mode
		switch (TranslationMode)
		{

			// Move
			case tmMove:
			{
				btnTranslationMode->setText("Move");
			}
			break;

			// Scale
			case tmScale:
			{
				btnTranslationMode->setText("Scale");
			}
			break;

			// Rotated
			case tmRotate:
			{
				btnTranslationMode->setText("Rotate");
			}
			break;

		};

	};

};

/*! Die Methode \b setDeformationMode() ndert den
 *  Deformationsalgorithmus des Effekts und aktualisiert die
 *  zugehrigen Steuerelemente.
 *  Eine bestehende Kontrollpunktselektion wird zurckgesetzt.
 *  Im Falle von B-Spline-Deformation wird geprft, ob
 *  die Kontrollgitterauflsung zulssig ist, und danach angepat,
 *  wenn ntig.
 *
 *  \note
 *  Der Parameter \b Mode sollte ein Wert der Enumeration \b enmDeformationMode sein.
 */
void CBSplineWrapperEffect::setDeformationMode(int Mode)
{

	if (DeformationMode!=(enmDeformationMode) Mode)
	{

		// Unselect all points first
		unselectAll();

		// Check if control mesh size is smaller then the b-spline order
		if (Mode == dmBSpline)
		{

			int newWidth  = ControlMesh->width();
			int newHeight = ControlMesh->height();

			if (newWidth<CurveOrder) newWidth = CurveOrder;
			if (newHeight<CurveOrder) newHeight= CurveOrder;

			if ( (newWidth!=ControlMesh->width()) || (newHeight!=ControlMesh->height()))
			{
				resizeControlMesh(newWidth,newHeight);
			};

		};

		// Set Deformation mode
		DeformationMode = (enmDeformationMode) Mode;

		// Set button caption in depenence of the selected mode
		switch (DeformationMode)
		{

			// Linear deformation
			case dmLinear:
			{

				// set button caption
				btnDeformationMode->setText("Linear");
				grpOrder->setEnabled(false);
				spnOrder->setEnabled(false);

				// set active deformer
				setActiveDeformer(Deformers[0]);

				// disable knot matrix table and reset it
				enableKnotMatrixPanel(false);

				// release control mesh size limit
				spnHResCMesh->setMinValue(2);
				spnVResCMesh->setMinValue(2);

				// Update view
				update();

			}
			break;

			// B-Spline deformation
			case dmBSpline:
			{

				// set button caption
				btnDeformationMode->setText("B-Spline");
				grpOrder->setEnabled(true);
				spnOrder->setEnabled(true);

				// set active deformer
				setActiveDeformer(Deformers[1]);

				// enable knot matrix table and fetch the active knot matrix
				enableKnotMatrixPanel(true);

				// Limit control mesh size
				spnHResCMesh->setMinValue(CurveOrder);
				spnVResCMesh->setMinValue(CurveOrder);

				// update view
				update();

			}
			break;

		};

	};

};

/*! Die Methode \b setEffectMode() ndert den
 *  Effektmodus und aktualisiert die zugehrigen Steuerelemente.
 *  Der jeweilige Effektmodus wird mit seinen Standardwerten initialisiert.
 *  Im Falle von Keyframe-Interpolation wird eine Keyframenavigationsleiste
 *  eingeblendet.
 *  Im Falle von Morphing wird eine Morphingnavigationsleiste angezeigt.
 *  Bevor in den Morphingmodus geschaltet wird, wird geprft, ob in jeder
 *  Bildliste des Frameworks mindestens ein Bild vorhanden ist. Ist dies der Fall,
 *  so erscheint eine Meldung und der Effektmodus wird nicht umgestellt.
 *
 *  \note
 *  Der Parameter \b Mode sollte ein Wert der Enumeration \b enmEffectMode sein.
 */
void CBSplineWrapperEffect::setEffectMode(int Mode)
{

	if (EffectMode!=(enmEffectMode) Mode)
	{

		// Set Effect mode
		EffectMode = (enmEffectMode) Mode;

		// Set button caption in depenence of the selected mode
		switch (EffectMode)
		{

			// Key frame mode
			case emKeyFrame:
			{

				btnEffectMode->setText("Key frame");

				// Hide morphing panel, show keyframe panel
				pnlMorphingNavigation->hide();
				pnlFrameNavigation->show();

				// Initialize Key frame mode
				initializeKeyFrameMode();

			}
			break;

			// Morphing mode
			case emMorphing:
			{
				// Check if both image lists contain at least one image
				if ( (parentFrame->getNumImages(0)>0) && (parentFrame->getNumImages(1)>0))
				{

					btnEffectMode->setText("Morphing");

					// Show morphing panel, hide keyframe panel
					pnlMorphingNavigation->show();
					pnlFrameNavigation->hide();

					// Initialize morphing mode
					initializeMorphingMode();

				}
				else
				{
					// Show message box
					QMessageBox mb("Error...","To use the morphing mode you have to specify at least one image for every image list!",QMessageBox::Warning,QMessageBox::Ok,0,0,dlgToolWindow);
					mb.exec();

					// Reset to keyframe mode
					EffectMode = emKeyFrame;
					btnEffectMode->setText("Key frame");

					// Hide morphing panel, show keyframe panel
					pnlMorphingNavigation->hide();
					pnlFrameNavigation->show();

				}
			}
			break;

		};

		// Update view
		update();

	};

};

/*! Die Methode \b insertKeyFrame() erzeugt einen Keyframe anhand
 *  des aktuellen Kontrollgitters. Das Keyframe wird an der
 *  Stelle eingefgt, an der sich der Framecursor befindet.
 *  Der Cursor wird um \b CURSOR_STEP Zeiteinheiten nach rechts verschoben.
 *  Die Zeitskalenansicht wird aktualisiert.
 */
void CBSplineWrapperEffect::insertKeyFrame()
{
	// Check first if effect is in keyframe mode
	if (EffectMode == emKeyFrame)
	{

		// Now make a copy of the current control mesh state and insert it
		// at the current frame cursor position
		KeyFrameList.insert(
			new CVertexGridKeyFrame<2,GLfloat>(usrKeyFrameBar->cursorPosition(),
				new CGrid< CVector<2, GLfloat> >(*ControlMesh)));
		usrKeyFrameBar->update();

		// Select the inserted frame
		usrKeyFrameBar->setSelectedKeyFrame(KeyFrameList.current(usrKeyFrameBar->cursorPosition()));

		// Move the key frame cursor to next insert position
		usrKeyFrameBar->setCursorPosition(usrKeyFrameBar->cursorPosition()+CURSOR_STEP);

		// Center view at cursor position
		usrKeyFrameBar->centerView();

		// Lock the control mesh resolution interface
		enableControlMeshResolutionPanel(false);

	};

};

/*! Die Methode \b deleteKeyFrame() lscht den gerade selektierten Keyframe.
 *  Die Zeitskalenansicht wird aktualisiert.
 */
void CBSplineWrapperEffect::deleteKeyFrame()
{

	// Check if any frame is selected
	if (&usrKeyFrameBar->selectedKeyFrame() !=NULL)
	{
		// Remove selected frame from list
		KeyFrameList.remove(usrKeyFrameBar->selectedKeyFrame().timeIndex());
		// Reset selection
		usrKeyFrameBar->setSelectedKeyFrame(NULL);
	};

	// Reenable control mesh resolution panel, if all key frames have been
	// deleted
	if (KeyFrameList.count() == 0) enableControlMeshResolutionPanel(true);

};

/*! Die Methode \b gotoZero() verschiebt den Framecursor an die Position 0.0.
 *  Die Zeitskalenansicht wird aktualisiert.
 */
void CBSplineWrapperEffect::gotoZero()
{

	// Cursorposition zurcksetzen
	usrKeyFrameBar->setCursorPosition(0);

	// View auf cursor zentrieren
	usrKeyFrameBar->centerView();

};

/*! Die Methode \b gotoFirstKeyFrame() setzt den  Framecursor auf das erste Keyframe.
 *  Das Keyframe wird selektiert.
 *  Die Zeitskalenansicht wird aktualisiert.
 */
void CBSplineWrapperEffect::gotoFirstKeyFrame()
{
	// Zum ersten Frame in der Liste gehen
	KeyFrameList.first();
	// Key frame leiste aktualisieren
	usrKeyFrameBar->setSelectedKeyFrame(KeyFrameList.current());

	if (KeyFrameList.current()!=NULL)
	{
		// Set the current control mesh state to the key frame data
		*ControlMesh = ((CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.current())->VertexGrid();
		// Cursor neu positionieren
		usrKeyFrameBar->setCursorPosition(KeyFrameList.current()->timeIndex());
		usrKeyFrameBar->centerView();
		// Update view
		update();
	};

};

/*! Die Methode \b gotoLastKeyFrame() setzt den  Framecursor auf das letzte Keyframe.
 *  Das Keyframe wird selektiert.
 *  Die Zeitskalenansicht wird aktualisiert.
 */
void CBSplineWrapperEffect::gotoLastKeyFrame()
{
	// Zum letzten Frame in der Liste gehen
	KeyFrameList.last();
	// Key frame leiste aktualisieren
	usrKeyFrameBar->setSelectedKeyFrame(KeyFrameList.current());

	if (KeyFrameList.current()!=NULL)
	{
		// Set the current control mesh state to the key frame data
		*ControlMesh = ((CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.current())->VertexGrid();
		// Cursor neu positionieren
		usrKeyFrameBar->setCursorPosition(KeyFrameList.current()->timeIndex());
		usrKeyFrameBar->centerView();
		// Update view
		update();
	};

};

/*! Die Methode \b gotoNextKeyFrame() setzt den Framecursor auf das Keyframe, welches
 *  direkt nach dem gerade selektierten Keyframe folgt.
 *  Das neue Keyframe wird selektiert.
 *  Die Zeitskalenansicht wird aktualisiert.
 *
 *  Ist kein Keyframe selektiert gewesen, so zeigt die Methode keinerlei Wirkung.
 */
void CBSplineWrapperEffect::gotoNextKeyFrame()
{
	// Zum nchsten Frame in der Liste gehen
	if (&usrKeyFrameBar->selectedKeyFrame()!=NULL)
	{

		KeyFrameList.current(usrKeyFrameBar->selectedKeyFrame().timeIndex());
		KeyFrameList.next();

		if (KeyFrameList.current()!=NULL)
		{

			// Key frame leiste aktualisieren
			usrKeyFrameBar->setSelectedKeyFrame(KeyFrameList.current());

			if (KeyFrameList.current()!=NULL)
			{
				// Set the current control mesh state to the key frame data
				*ControlMesh = ((CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.current())->VertexGrid();
				// Cursor neu positionieren
				usrKeyFrameBar->setCursorPosition(KeyFrameList.current()->timeIndex());
				usrKeyFrameBar->centerView();
				// Update view
				update();
			};

		};

	};

};

/*! Die Methode \b gotoNextKeyFrame() setzt den Framecursor auf das Keyframe, welches
 *  dem gerade selektierten Keyframe vorangeht.
 *  Das neue Keyframe wird selektiert.
 *  Die Zeitskalenansicht wird aktualisiert.
 *
 *  Ist kein Keyframe selektiert gewesen, so zeigt die Methode keinerlei Wirkung.
 */
void CBSplineWrapperEffect::gotoPreviousKeyFrame()
{
	// Zum vorherigen Frame in der Liste gehen
	if (&usrKeyFrameBar->selectedKeyFrame()!=NULL)
	{

		KeyFrameList.current(usrKeyFrameBar->selectedKeyFrame().timeIndex());
		KeyFrameList.previous();

		if (KeyFrameList.current()!=NULL)
		{
			// Key frame leiste aktualisieren
			usrKeyFrameBar->setSelectedKeyFrame(KeyFrameList.current());

			if (KeyFrameList.current()!=NULL)
			{
				// Set the current control mesh state to the key frame data
				*ControlMesh = ((CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.current())->VertexGrid();
				// Cursor neu positionieren
				usrKeyFrameBar->setCursorPosition(KeyFrameList.current()->timeIndex());
				usrKeyFrameBar->centerView();
				// Update view
				update();
			};

		};

	};

};

/*! Die Methode \b controlMeshToSource() kopiert das aktuelle
 *  Kontrollgitter in das Morphingquellgitter.
 *  Das aktuelle Arbeitstexelmesh wird ebenfalls dem Morphingquelltexelmesh zugewiesen.
 *  Der Effekt wird anschlieend neu gerendert.
 */
void CBSplineWrapperEffect::controlMeshToSource()
{

	if (EffectMode==emMorphing)
	{

		// Recalculate texel mesh
		recalculateTexelCoordinates(*ImageMesh,*TexelMesh);

		// Copy current control mesh to first key frame
		((CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.first())->VertexGrid() = *ControlMesh;

		// Copy texel mesh
		*TexelMeshA = *TexelMesh;

		// Set morphing edit state
		MorphingEditState = esSource;

	};

};

/*! Die Methode \b controlMeshToDestination() kopiert das aktuelle
 *  Kontrollgitter in das Morphingzielgitter.
 *  Das aktuelle Arbeitstexelmesh wird ebenfalls dem Morphingzieltexelmesh zugewiesen.
 *  Der Effekt wird anschlieend neu gerendert.
 */
void CBSplineWrapperEffect::controlMeshToDestination()
{

	if (EffectMode==emMorphing)
	{

		// Recalculate texel mesh
		recalculateTexelCoordinates(*ImageMesh,*TexelMesh);

		// Copy current control mesh to first key frame
		((CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.last())->VertexGrid() = *ControlMesh;

		// Copy texel mesh
		*TexelMeshB = *TexelMesh;

		// Set morphing edit state
		MorphingEditState = esDestination;

	};

};

/*! Die Methode \b sourceToControlMesh() kopiert das Morphingquellkontrollgitter
 *  in das aktuelle Kontrollgitter.
 *  Das Morphingquelltexelmesh wird dem Arbeitstexelmesh zugewiesen.
 *  Ferner wird das erste Bild der Quellmorphingbildsequenz aktiviert.
 *  Der Effekt wird anschlieend neu gerendert.
 */
void CBSplineWrapperEffect::sourceToControlMesh()
{

	// Check if morphing is active
	if (EffectMode==emMorphing)
	{

		// Reposition cursor
		usrKeyFrameBar->setCursorPosition(0.0f);
		MorphingTimeIndex =0.0;

		// Copy control mesh
		*ControlMesh=((CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.first())->VertexGrid();

		// Get texel mesh
		*TexelMesh = *TexelMeshA;

		// Deform
		ActiveDeformer->deform();

		// Get first image from source frame list
		texImage1 = parentFrame->fetchImage(0,0);

		// Update view
		TranslationState = tsNothing;
		update();

		// Set morphing edit state
		MorphingEditState = esSource;

	};

};

/*! Die Methode \b destinationToControlMesh() kopiert das Morphingzielkontrollgitter
 *  in das aktuelle Kontrollgitter.
 *  Das Morphingzieltexelmesh wird dem Arbeitstexelmesh zugewiesen.
 *  Ferner wird das letzte Bild der Zielmorphingbildsequenz aktiviert.
 *  Der Effekt wird anschlieend neu gerendert.
 */
void CBSplineWrapperEffect::destinationToControlMesh()
{

	// Check if morphing is active
	if (EffectMode==emMorphing)
	{

		// Reposition cursor
		usrKeyFrameBar->setCursorPosition(1.0f);
		MorphingTimeIndex =1.0;

		// Copy control mesh
		*ControlMesh=((CVertexGridKeyFrame<2,GLfloat>*) KeyFrameList.last())->VertexGrid();

		// Get texel mesh
		*TexelMesh = *TexelMeshB;

		// Deform
		ActiveDeformer->deform();

		// Get last image from destination frame list
		texImage1 = parentFrame->fetchImage(1,parentFrame->getNumImages(1)-1);

		// Update view
		TranslationState = tsNothing;
		update();

		// Set morphing edit state
		MorphingEditState = esDestination;

	};

};

/*! Die Methode \b clearKeyFrames() lscht alle Keyframes aus
 *  der Keyframeliste. Die Zeitskalenansicht wird danach aktualisiert.
 *  Der Benutzer wird vorher gefragt, ob er wirklich alle Keyframes
 *  lschen mchte.
 */
void CBSplineWrapperEffect::clearKeyFrames()
{
	if (KeyFrameList.count()>0)
	{
		// Create message box
		QMessageBox mb("Clear...","Are you sure you want to delete all key frames from list?",
									 QMessageBox::Warning ,
									 QMessageBox::Yes | QMessageBox::Default,
									 QMessageBox::No  | QMessageBox::Escape ,0,dlgToolWindow);

		if (mb.exec() == QMessageBox::Yes)
		{
			// Clear frame list
			KeyFrameList.clear();

			// Move cursor to zero
			gotoZero();

			// Enable control grid resultion panel
			if (EffectMode == emKeyFrame)
			{
				enableControlMeshResolutionPanel(true);
			}

		};
	};

};

/*! Die Methode \b resetControlMesh() setzt das Kontrollgitter zurck
 *  auf ein regulres rechteckiges Gitter.
 *  Der Effekt wird anschlieend gerendert.
 */
void CBSplineWrapperEffect::resetControlMesh()
{
	// Make control mesh regular
	makeRegularMesh(ControlMesh,-1.0f,-1.0f,2.0f,2.0f);

	// Deform image mesh
	if (ActiveDeformer!=NULL)
	{
		ActiveDeformer->deform();
	};

	// Update view
	update();
};

/*! Die Methode \b updateKnotMatrix() kopiert das durch
 *  \b row und \b col spezifizierte Element aus dem
 *  Tabellenwidget, in die gerade aktive Knotenmatrix.
 *  Es werden zuvor einige Plausibilitts- und Bereichsprfungen
 *  durchgefhrt. Falls diese Fehlschlagen, so wird der Benutzer
 *  darber informiert, und die nderung rckgngig gemacht.
 *  Danach wird der Effekt neu gerendert.
 */
void CBSplineWrapperEffect::updateKnotMatrixValue(int row, int col)
{

	// Get matrix
	CGrid<float>* matrix;

	switch(ActiveKnotMatrix)
	{

		case kmRow:
			matrix = &((CBSplineDeformer<GLfloat>*) ActiveDeformer)->rowKnotMatrix();
		break;

		case kmCol:
			matrix = &((CBSplineDeformer<GLfloat>*) ActiveDeformer)->colKnotMatrix();
		break;

	};

	// Get value as string
	QString strVal(tabKnotMatrix->text(row,col));
	bool valOK;

	// Convert string to float
	float fltVal = strVal.toFloat(&valOK);
	// Check if conversion was successful
	if (valOK)
	{
		// check value range
		if ((fltVal>=0.0f) && (fltVal<=1.0f))
		{

			// update only if value differs
			if (fabsf(fltVal-matrix->get(col,row)) > 1E-8)
			{
				// Copy value to matrix
				matrix->set(col,(matrix->height()-1)-row,fltVal);
				// Update view
				update();

				// Check if morphing mode is active
				if (EffectMode == emMorphing)
				{
					recalculateMorphingTexelCoordinates();
				}

			};

		}
		else
		{
			// Show message box
			QMessageBox::warning(dlgToolWindow,"Range error...","The value you enter must be between 0.0 and 1.0");
			tabKnotMatrix->setText(row,col,QString::number(matrix->get(col,(matrix->height()-1)-row)));
		}

	}
	else
	{
		tabKnotMatrix->setText(row,col,QString::number(matrix->get(col,row)));
	};

};

/*! Die Methode \b resetKnotMatrix() setzt beide
 *  Knotenmatrizen auf uniforme und normalisierte Matrizen
 *  zurck. Hierzu wird die Methode \b buildUniformNodeMatrices()
 *  der Klasse \b CBSplineDeformer benutzt.
 *  Das Tabellenwidget wird mittels \b knotMatrixToTable() aktualisiert.
 *  Der Effekt wird anschlieend neu gerendert.
 */
void CBSplineWrapperEffect::resetKnotMatrix()
{

	if (DeformationMode == dmBSpline)
	{
		// Rebuild node matrices
		((CBSplineDeformer<GLfloat>*) ActiveDeformer)->buildUniformNodeMatrices();

		// Update matrix table
		knotMatrixToTable(ActiveKnotMatrix);

		// Update view
		update();

	};

};

/*! Die Methode \b setActiveKnotMatrix() setzt die
 *  zu bearbeitende Knotenmatrix.
 *  Dabei wird der Inhalt des Tabellewidgets entsprechend
 *  aktualisiert.
 *
 *  \note
 *  Der Parameter \b Matrix sollte ein Wert aus der Enumeration \b enmKnotMatrixType sein.
 */
void CBSplineWrapperEffect::setActiveKnotMatrix(int Matrix)
{
	ActiveKnotMatrix = (enmKnotMatrixType) Matrix;

	// Update button caption

	switch (ActiveKnotMatrix)
	{
		case kmRow:
		{
			btnMatrixType->setText("Row matrix");
		}
		break;

		case kmCol:
		{
			btnMatrixType->setText("Col matrix");
		}
		break;

	};

	// Update matrix
	if (DeformationMode == dmBSpline)
	{
		knotMatrixToTable(ActiveKnotMatrix);
	};

};



