/*
    Copyright  2001 Christoph Brzozowski - All Rights Reserved.
 
 	This file is part of GL Effects Framework.
 
    GL Effects Framework is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    GL Effects Framework is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GL Effects Framework; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/*! @file CBSplineDeformer.h
 *
 *  @brief
 *	Deklarationen der Klasse \b CBSplineDeformer.
 *   
 *  @author Christoph Brzozowski.
 *  
 */

#ifndef cls_CBSplineDeformerH
#define cls_CBSplineDeformerH

#if     _MSC_VER > 1000
#pragma once
#endif

#include <stdlib.h>
#include <stdexcept>
using std::invalid_argument;

#include "CVector.h"
#include "CGrid.h"
#include "CDynamicArray.h"
#include "SplineLib.h"


/*!
 *  @class CBSplineDeformer
 *
 * @ingroup bsplineWarpClasses
 * 
 * \brief Klassentemplate, welches einen Deformationsalgorithmus implementiert, der
 *  auf Vertexgittern operiert und B-Spline-Interpolation nutzt.
 * 
 *  Das Klassentemplate \b CBSplineDeformer implementiert einen linearen
 *  Deformationsalgorithmus, der Vertexgitter mit Hilfe eines Kontrollgitters
 *  verzerrt. Zur berechnung der Vertexkoordinaten wird B-Spline-Interpolation
 *  verwendet.
 *
 *  Die Klasse erlaubt die Spezifikation eines Quell-, eines Ziel- und eines Kontrolgitters.
 *  Die Elemente der Vertexgrids sind vom Typ \b CVector. Die Anzahl der Komponenten wird 
 *  vom Template auf 2 festgesetzt. 
 *  Der Templateparameter \b T gibt den Datentyp der Vektorkomponenten an.
 *  Ferner knnen fr die B-Spline-Interpolation spezifischen Parameter, wie Funktionsordnung und 
 *  Knotenvektoren spezifiziert werden.
 *
 *  \remarks
 *  Das als Quell- und Zielgitter kann ein und das selbe Gitter spezifiziert werden, da
 *  der Deformationsalgorithmus eine Kopie der Gitterkoordinaten anlegt, und danach
 *  auf die Originalkoordinaten des Quellgitters nicht mehr zugreift.
 *
 *  \note Die Ausmae der zugewiesenen Gitter mssen bereinstimmen, da ansonsten
 *        eine Exception vom Typ \b invalid_argument ausgelst wird.
 *        Ausgenommen hiervon ist das Kontrollgitter.
 *
 *  \note Die Mae der Grids sollten mindestens 2 x 2 betragen.
 *
 *  \note Nachdem alle drei Gitter spezifiziert wurden, hat ein verndern
 *        der Positionen der Quellgittervertices keine Auswirkungen auf das
 *        Ergebnis.
 *        Um den gewnschten Effekt zu erhalten, mssen die Gitter nochmals
 *        neu zugewiesen werden. 
 *
 *  \pre
 *  Fr den Datentyp \b T mssen alle gngigen arithmetischen Operatoren berladen sein.
 *  Ferner sollte der \b T zu \b float oder zu \b int kompatibel sein.
 *
 */
template <class T> class CBSplineDeformer : public CControlGridDeformer<2,T>
{

protected:

	//! Speichert die Kurvenordnung.
	int CurveOrder; 

	//! Speichert die Vertex-Koordinaten des Quellgitters relativ zum Kontrollgitter.
	CDynamicArray< CVector<2,float> > CellCoordinates; 


	//! Speichert die Knotenvektorindizes der Quellgitter-Vertices und den Zielindex innerhalb des Zielgrids.
	CDynamicArray< CVector<3,int> > IndexCoordinates;

	//! Speichert temporre Kontrollpunkte.
	CDynamicArray< CVector<2,T> > ControlVertices;

	//! Speichert die Zeilenknotenvektoren.
	CGrid<float> RowNodeMatrix;

	//! Speichert die Spaltenknotenvektoren.
	CGrid<float> ColNodeMatrix;

	//! Berechnet die Koordinaten der Quellgittervertices relativ zu Kontrollgitter und erzeugt die Zielindextabelle.
	void buildCellArray();

	//! Macht nichts...
	void buildIndexArray();

	//! ndert die Gre der Knotenmatrizen unter Bercksichtigung aller Parameter.
	void resizeNodeMatrices();

	//! Gibt den Index des Knotenvektorintervalls, in welchem sich \b u befindet.
	int getIntervalIndex(float u, int KnotVectorSize, float* KnotVector);

public:

	//! Standardkonstruktor.
	CBSplineDeformer();

	//! Gibt die Kurvenordnung zurck.
	int order()
	{

		return CurveOrder;

	};

	//! Gibt eine Referenz auf die Spaltenknotenmatrix zurck.
	CGrid<float>& colKnotMatrix()
	{

		return ColNodeMatrix;

	};

	//! Gibt eine Referenz auf die Zeilenknotenmatrix zurck.
	CGrid<float>& rowKnotMatrix()
	{

		return RowNodeMatrix;

	};

	//! Setzt die Kurvenordnung.
	void setOrder(int newOrder);

	//! Setzt die Referenz auf das Quellgitter.
	virtual void setSourceGrid(CGrid< CVector<2,T> >* objSourceGrid);

	//! Setzt die Rerefenz auf das Kontrollgitter.
	virtual void setControlGrid(CGrid< CVector<2,T> >* objControlGrid);
	
	//! Erzeugt uniforme Knotenmatrizen.
	void buildUniformNodeMatrices();
	
	//! Implementiert den eigentlichen Deformationsalgorithmus.
	virtual void deform();

	//! Standarddestruktor.
	virtual ~CBSplineDeformer();

};

/*! Der Standardkonstruktor setzt alle dynamischen Arrays auf die Gre 0 zurck.
 *  Ebenfalls die Kurvenordnung.
 */
template <class T> CBSplineDeformer<T>::CBSplineDeformer() : CControlGridDeformer<2,T>()
{

	// Dynamische Arrays zurcksetzen
	CellCoordinates.resize(0);
	IndexCoordinates.resize(0);
	ControlVertices.resize(0);
	RowNodeMatrix.resize(0,0);
	ColNodeMatrix.resize(0,0);

	// Kurvenordnung setzen
	CurveOrder = 0;

};

/*! Die Methode \b buildCellArray() erzeugt eine Liste aller
 *  Vertices, die sich innerhalb des Kontrollgitters befinden.
 *  Ferner werden ihre zum Kontrollgitter relativen Koordinaten
 *  in einem weiteren Array gespeichert.
 *  Desweiteren erzeugt die Routine eine Liste mit den 
 *  Indizes der zu deformierenden Vertices innerhalb des Zielgitters.
 *
 *  \pre Das Kontrollgitter muss regelmig sein, d.h. aus
 *       Rechtecken bestehen, damit der Algorithmus
 *       korrekt funktioniert.
 *
 */
template <class T> void CBSplineDeformer<T>::buildCellArray()
{

	// Vertexzhler
	int count = 0;
	// Schleifenzhler
	int i = 0;

	// Anzahl der Vertices innerhalb des Kontrollgitters zhlen
	CVector<2,T>* Elements = (CVector<2,T>*) SourceGrid->getElements();
	for (i=0; i<SourceGrid->width()*SourceGrid->height(); i++)
	{

		// Prfen, ob Vertex innerhalb des Kontrollgitters liegt
		if (
			  (Elements->c[0]>=ControlGrid->get(0,0).c[0]) &&
				(Elements->c[0]<=ControlGrid->get(ControlGrid->width()-1,0).c[0]) &&
				(Elements->c[1]>=ControlGrid->get(0,0).c[1]) &&
				(Elements->c[1]<=ControlGrid->get(0,ControlGrid->height()-1).c[1])
			 )
		{

			count++;

		}

		Elements++;

	};

	// Zellkoordinatenarray neu dimensionieren
	CellCoordinates.resize(count);

	// Knotenindexarray neu dimensionieren
	IndexCoordinates.resize(count);
	
	// Koordinaten der Vertices relativ zum Kontrollgitter berechnen
	Elements = (CVector<2,T>*) SourceGrid->getElements();
	CVector<2,float>* DestElements = CellCoordinates.get();
	CVector<3,int>* DestIndices = IndexCoordinates.get();

	// Gittermae ermitteln
	float GridWidth  = ControlGrid->get(ControlGrid->width()-1,0).c[0]-ControlGrid->get(0,0).c[0];
	float GridHeight = ControlGrid->get(0,ControlGrid->height()-1).c[1]-ControlGrid->get(0,0).c[1];
	CVector<2,T> GridOrigin = ControlGrid->get(0,0);

	for (i=0; i<SourceGrid->width()*SourceGrid->height(); i++)
	{
		
		// Prfen, ob Vertex innerhalb des Kontrollgitters liegt
		if (
			  (Elements->c[0]>=ControlGrid->get(0,0).c[0]) &&
				(Elements->c[0]<=ControlGrid->get(ControlGrid->width()-1,0).c[0]) &&
				(Elements->c[1]>=ControlGrid->get(0,0).c[1]) &&
				(Elements->c[1]<=ControlGrid->get(0,ControlGrid->height()-1).c[1])
			 )
		{

			// Relative Koordinaten speichern
			DestElements->c[0] = RowNodeMatrix.get(RowNodeMatrix.width()-1,0)*(Elements->c[0]-GridOrigin.c[0])/(GridWidth);
			DestElements->c[1] = ColNodeMatrix.get(ColNodeMatrix.width()-1,0)*(Elements->c[1]-GridOrigin.c[1])/(GridHeight);

			// Zielindex speichern
			DestIndices->c[2] = i;

			// In den Ziellisten weiter voranschreiten
			DestElements++;
			DestIndices++;

		}

		// In der Quelliste weiter voranschreiten
		Elements++;

	};

};

/*! 
 * Die Methode macht nichts, auer das Indexarray neu zu dimensionieren.
 */
template <class T> void CBSplineDeformer<T>::buildIndexArray()
{

	// Arraygre neu dimensionieren
	IndexCoordinates.resize(CellCoordinates.getSize());

};

/*! Die Methode \b resizeNodeMatrices() redimensioniert die Gre der Knotenmatrizen unter 
 *  Bercksichtigung der Kurvenordnung und er Kontrollgitterausmae.
 *  Die Zeilenmatrix hat die Ausmae (\b Kontrollgitterbreite + \b Kurvenordnung) x \b Kontrollgitterhhe.
 *  Die Spaltenmatrix hat die Ausmae (\b Kontrollgitterhhe + \b Kurvenordnung) x 1.
 */
template <class T> void CBSplineDeformer<T>::resizeNodeMatrices()
{

	// Prfen, ob das Kontrollgitter bereits zugewiesen wurde
	if (ControlGrid!=NULL)
	{

		// Matrizen neudimensionieren
		RowNodeMatrix.resize(ControlGrid->width()+CurveOrder,ControlGrid->height());
		ColNodeMatrix.resize(ControlGrid->height()+CurveOrder,1);

	}
	else
	{

		// Knotenmatrizen zurcksetzen
		RowNodeMatrix.resize(0,0);
		ColNodeMatrix.resize(0,0);

	};

};

/*! Die Methode \b buildUniformMatrices() initialisiert die Spalten- und Zeilenknotenmatrix
 *  mit uniformen normierten Knotenvektoren.
 *  Die ersten \b CurveOrder+1 Werte des Knotenvektors sind 0.0,
 *  die letzten \b CurveOrder+1 Werte des Knotenvektors sind 1.0.
 *  Die brigen Werte werden zwischen 0 und 1 linear interpoliert.
 */
template <class T> void CBSplineDeformer<T>::buildUniformNodeMatrices()
{

	int x,y,n;    // Schleifen- und Indexzhler
	float t;      // Zeitindex

	// Zeilenmatrix berechnen
	for (y=0 ; y<RowNodeMatrix.height() ; y++)
	{

		n = 0;
		t = 0.0f;

		// Die ersten <CurveOrder> Elemente auffllen
		for (x=0 ; x<CurveOrder ; x++)
		{

			RowNodeMatrix.set(n,y,t);
			n++;

		};

		t+=1.0f;

		// Mittelelemente auffllen
		for (x=0 ; x<RowNodeMatrix.width()-(CurveOrder)*2 ; x++)
		{

			RowNodeMatrix.set(n,y,t);
			n++;
			t+=1.0f;

		}
		
		// Die letzten <CurveOrder> Elemente auffllen
		for (x=0 ; x<CurveOrder ; x++)
		{

			RowNodeMatrix.set(n,y,t);
			n++;

		};

		// Vektor normalisieren
		for (x=0; x<RowNodeMatrix.width() ; x++)
		{
			
			float r = RowNodeMatrix.get(x,y)/RowNodeMatrix.get(RowNodeMatrix.width()-1,y);
			RowNodeMatrix.set(x,y,r);

		};

	};

	// Spaltenmatrix berechnen
	for (y=0 ; y<ColNodeMatrix.height() ; y++)
	{

		n = 0;
		t = 0.0f;

		// Die ersten <CurveOrder> Elemente auffllen
		for (x=0 ; x<CurveOrder ; x++)
		{
			ColNodeMatrix.set(n,y,t);
			n++;
		};

		t+=1.0f;

		// Mittelelemente auffllen
		for (x=0 ; x<ColNodeMatrix.width()-(CurveOrder)*2 ; x++)
		{
			ColNodeMatrix.set(n,y,t);
			n++;
			t+=1.0f;
		}
		
		// Die letzten <CurveOrder> Elemente auffllen
		for (x=0 ; x<CurveOrder ; x++)
		{
			ColNodeMatrix.set(n,y,t);
			n++;
		};

		// Vektor normalisieren
		for (x=0; x<ColNodeMatrix.width() ; x++)
		{
			float r = ColNodeMatrix.get(x,y)/ColNodeMatrix.get(ColNodeMatrix.width()-1,y);
			ColNodeMatrix.set(x,y,r);
		};

	};

};

/*! Die Methode \b setOrder() setzt die Ordnung der 
 *  B-Spline-Kurven. 
 * 
 *  \note
 *  Wurde bereits ein Kontrollgitter zugewiesen, so werden die
 *  Knotenmatrizen neudimensioniert und mit uniformen normalisierten
 *  Knotenvektoren initialisiert.
 *
 */
template <class T> void CBSplineDeformer<T>::setOrder(int newOrder)
{
	// Prfen, ob sich die Ordnung von der bisherigen Unterscheidet

	if (newOrder!=CurveOrder)
	{
		
		// Kurvenordnung setzen
		CurveOrder = newOrder;

		// Prfen, ob bereits ein Kontrollgitter zugewiesen wurden
		if (ControlGrid!=NULL)
		{
			
			// Knotenmatrizen neudimensionieren
			resizeNodeMatrices();
			// Knotenmatrizen mit Uniformen Knotenvektoren fllen
			buildUniformNodeMatrices();

			// Prfen, ob bereits ein Quellgitter zugewiesen wurde
			if (SourceGrid!=NULL)
			{		
				// Knotenindexarray neu erstellen
				buildIndexArray();
			};

		};

	};

};

/* Die Methode \b setSourceGrid() weist dem Deformationsalgorithmus ein Quellgitter zu.
 * Wurde bereits ein Kontrollgitter spezifiziert so wird die Methode
 * \b buildCellArray() aufgerufen, um die Liste aller sich im Kontrollgitter
 * befindenden Quellvertices zu erzeugen. 
 *
 * \note Es wird eine Exception vom Typ \b invalid_argument ausgelst, falls
 *       die Ausmae des Quellgitters nicht mit den Ausmaen des Zielgitters bereinstimmen.
 */
template <class T> void CBSplineDeformer<T>::setSourceGrid(CGrid< CVector<2,T> >* objSourceGrid)
{

	// Quellgitter zuweisen
	CControlGridDeformer<2,T>::setSourceGrid(objSourceGrid);

	// Prfen, ob das Quellgitter und das Kontrollgitter zugewiesen wurden
	if ((SourceGrid!=NULL) && (ControlGrid!=NULL))
	{
			// Vertexkoordinaten relativ zum Kontrollgitter berechnen
			buildCellArray();
			// Knotenindexarray neu erstellen
			buildIndexArray();
	};
	
};

// Weist ein Kontrollgitter zu. Falls bereits ein Quellgitter 
// zugewiesen wurde wird eine Vertexzugehrigkeitstabelle erzeugt.
// Ferner werden die Ausmae der Knotenmatrizen angepasst und diese mit uniformen
// Knotenvektoren gefllt.

/* Die Methode \b setControlGrid() weist dem Deformationsalgorithmus ein Kontrollgitter zu.
 * Wurde bereits ein Quellgitter spezifiziert so wird die Methode
 * \b buildCellArray() aufgerufen, um die Liste aller sich im Kontrollgitter
 * befindenden Quellvertices zu erzeugen. 
 * Ferner werden die Ausmae der Knotenmatrizen mittels \b resizeNodeMatrices() angepat.
 */
template <class T> void CBSplineDeformer<T>::setControlGrid(CGrid< CVector<2,T> >* objControlGrid)
{
	
	// Prfen, ob ein Kontrollgitter angegeben wurde
	if (objControlGrid!=NULL)
	{

		// Prfen, ob bereits ein Quellgitter zugewiesen wurde
		if (SourceGrid!=NULL)
		{

			// Prfen, ob zuvor schon ein Kontrollgitter zugewiesen wurde und, ob
			// die Ausmae bereinstimmen
			if (
				  (ControlGrid==NULL) || 
				  ( 
					  (ControlGrid!=NULL) && 
						(
						 (ControlGrid->width()!=objControlGrid->width()) || 
					   (ControlGrid->height()!=objControlGrid->height()
						)
					 )
				  )
					)
			{
				
				// Kontrollgitter zuweisen
				CControlGridDeformer<2,T>::setControlGrid(objControlGrid);

				// Knotenmatrizen neudimensionieren
				resizeNodeMatrices();

				// Mit uniformen Vektoren fllen
				buildUniformNodeMatrices();

				// Vertexkoordinaten relativ zum Kontrollgitter berechnen
				buildCellArray();

				// Knoten- und Zielindizes berechnen
				buildIndexArray();

			}
			else
			{

				// Kontrollgitter zuweisen ohne Neuerstellung der Knotenmatrizen und
				// der Zugehrigen Index- und Zellenarrays, da die zuvor erstellten
				// weiterhin passen
				CControlGridDeformer<2,T>::setControlGrid(objControlGrid);

			};

		}
		else
		{

			// Kontrollgitter zuweisen ohne Erstellung der
			// Index- und Zellenarrays
			CControlGridDeformer<2,T>::setControlGrid(objControlGrid);

			// Knotenmatrizen neudimensionieren
			resizeNodeMatrices();

			// Mit uniformen Vektoren fllen
			buildUniformNodeMatrices();

		};

	}
	else
	{

		// Arrays zurcksetzen
		CellCoordinates.resize(0);
		IndexCoordinates.resize(0);
		ControlVertices.resize(0);
		RowNodeMatrix.resize(0,0);
		ColNodeMatrix.resize(0,0);

		// Kontrollgitter zuweisen
		CControlGridDeformer<2,T>::setControlGrid(objControlGrid);

	};
	
};

/*! Die Methode durchluft den ber \b KnotVector angegebenen Knotenvector, und berprft, ob
 *  sich \b u in einem der durch den Vektor definierten Intervalle befindet.
 *  Die Methode gibt den Index des ermittelten Intervalls zurck.
 */
template <class T> int CBSplineDeformer<T>::getIntervalIndex(float u, int KnotVectorSize, float* KnotVector)
{

	KnotVectorSize-=(CurveOrder-1);

	// Invervall suchen
	for (int i = CurveOrder-1 ; i<KnotVectorSize ; i++)
	{
		
		if ((u>=KnotVector[i]) && (u<KnotVector[i+1]))
		{
			// Gefundenes Intervall zurckgeben
			return i;
		};

	};

	// Letztes Intervall zurckgeben
	return KnotVectorSize+1;
	
};

// Deformiert das Quellgitter mittels B-Spline-Interpolation und speichert das
// Ergebnis im Zielgitter. Zur Berechnung der B-Spline-Werte wird der
// rekursive deBoor-Algorithmus verwendet.

/*! Die Methode \b deform() implementiert den eigentlichen Deformationsalgorithmus.
 *  Das Quellgitter wird mittels B-Spline-Interpolation verformt.
 *  Dabei wird mit Hilfe der Funktionen aus \b BSplineLib.h und \b BSplineLib.cpp
 *  Das Tensorprodukt der durch die Kontrollgitterpunkte definierten B-Splines
 *  berechnet.
 *  Das Ergebnis wird in dem Zielgitter gespeichert.
 *
 *  \remarks
 *  Da der Algorithmus nur die zwischengespeicherten relativen
 *  Koordinaten der Quellvertices benutzt, ist es durchaus
 *  erlaubt als Quellgitter und als Ziellgitter ein und das selbe
 *  Vertexgitter zu spezifizieren.
 *
 */
template <class T> void CBSplineDeformer<T>::deform()
{

	// Kontrollpunktearray neu dimensionieren
	ControlVertices.resize(ControlGrid->height());

	// Prfen, ob alle Grids angegeben wurden
	if ((SourceGrid!=NULL) && (ControlGrid!=NULL) && (DestinationGrid!=NULL))
	{

		// Alle Punkte im Zellenarray durchlaufen
		CVector<2,float>* Coordinates = CellCoordinates.get();
		CVector<2,float>* Destination = (CVector<2,float>*) DestinationGrid->getElements();
		CVector<3,int>* Indices = IndexCoordinates.get();

		// Basis-Matrix alloziieren
		CGrid<float> BaseMatrix(CurveOrder+2,CurveOrder+2);
		BaseMatrix.clear(0.0f);

		for(int i = 0 ; i<CellCoordinates.getSize() ; i++)
		{
			// Relative Koordinaten ermitteln
			float t = Coordinates->c[0];
			float v = Coordinates->c[1];

			// Zeiger auf die zugehrigen Knotenvektoren ermitteln
			float* RowNodeVector = (float*) RowNodeMatrix.getElements();
			float* ColNodeVector = (float*) ColNodeMatrix.getElements();

			// Zeiger auf die Kontrollvertexdaten ermitteln
			CVector<2,T>* ControlVertex = ControlVertices.get();

			// Ersten Teil des Tensorprodukts berechnen
			for (int y = 0; y<ControlGrid->height(); y++)
			{

				// Intervallindex des Vertex im y-ten Zeilenknotenvektor ermitteln
				int index = getIntervalIndex(t,RowNodeMatrix.width(),RowNodeVector+y*RowNodeMatrix.width());

				// Splinepunkt berechnen
				*ControlVertex = CalculateSplinePoint(t,CurveOrder,ControlGrid->width()-1,index,((CVector<2,T>*) ControlGrid->getElements())+y*ControlGrid->width(),RowNodeVector+y*RowNodeMatrix.width(),(float*) BaseMatrix.getElements());
				
				// Zum nchsten Kontrollvertex voranschreiten
				ControlVertex++;

			};

			// Zweiten Teil des Tensorprodukts berechnen und somit den Endpunkt
			// bestimmen.

			// Knotenvektorintervall ermitteln
			int index = getIntervalIndex(v,ColNodeMatrix.width(),ColNodeVector);
			// B-Spline-Punkt berechnen und als Ergebnis speichern
			Destination[Indices->c[2]] = CalculateSplinePoint(v,CurveOrder,ControlGrid->height()-1,index,ControlVertices.get(),ColNodeVector,(float*) BaseMatrix.getElements());

			// Zum nchsten Punkt
			Coordinates++;
			Indices++;

		};

	};

};

/*! Der Standarddestruktor gibt jeglichen dynamsich
 *  alloziierten Speicher wieder frei. Die zugewiesenen
 *  Gitter ausgenommen.
 */ 
template <class T> CBSplineDeformer<T>::~CBSplineDeformer()
{
	
	// Dynamische Arrays zurcksetzen
	CellCoordinates.resize(0);
	IndexCoordinates.resize(0);
	ControlVertices.resize(0);
	RowNodeMatrix.resize(0,0);
	ColNodeMatrix.resize(0,0);

};

#endif



