/*****************************************************************************\
 *                              AccumEffect
\*****************************************************************************/

/*! @file AccumEffect.cpp
 *
 *  @brief
 *	AccumEffect Effekt Klasse: benutzt den Accumulationbuffer fr Bildsubtraktion 
 *  und Motion Blur.
 *    
 *  @author 
 *  P.Claren & C.Mlhaupt
 *  
 */

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

// Qt headers
#include <qgl.h>

#include <qtimer.h>
#include <qgroupbox.h>


// Local Headers
#include "AccumEffectPanel.h"
#include "AccumEffect.h"

/*****************************************************************************\
 *                              AccumEffect
\*****************************************************************************/

// **** Construction/desctruction ****

/**
 * @param parent Pointer to the parent window.
 *  
 * Es werden folgende effectInfo Einstellungen gemacht:
 *
 * Es wird eine Bildliste bentigt (die Filmsequenz), 
 * needNewTextures = true, 
 * needRawTextures = false
 * 
 * @see GLEffect::GLEffect(), effectInfo
 **/

AccumEffect::AccumEffect(GLFramework* parent):GLEffect(parent)
{

	effectInfo->effectName = "Bildeffekte mit Accumulation Buffer";
	effectInfo->fileDialogNames[0] = "Load Sequenz";
	effectInfo->needNewTextures = true;
	effectInfo->requiredImageLists=1;
	effectInfo->shortDescription = "AccumEffect";
	effectInfo->version = "1.0";
	effectInfo->author = "C.Mlhaupt & P.Claren";
	effectInfo->needRawTextures = false;
	
	createControlPanel(true);
	hideControls();

}

/**
 * 
 * Setzt GL_TEXTURE_2D und GL_DEPTH_TEST zurck.
 * Lsche die Instanz der AccumEffectPanel Klasse.
 *  
 * @see GLEffect::~GLEffect() , AccumEffectPanel::~AccumEffectPanel()
 **/AccumEffect::~AccumEffect()
{	
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_DEPTH_TEST);
	delete cPan;
}

// Initialization
/**
 * 
 * Setzt die Parameter zur korrekten Dastellung des OpenGL-Kontextes.
 * Setzt Acc_curImg=0. Acc_curImg ist das aktuelle Bild der Sequenz.
 * Setzt Acc_lstLen auf die Anzahl der Bilder der Sequenz. Acc_lstLen ist die Listen Lnge
 *  
 * @see GLEffect::init()
 **/void AccumEffect::initialize() 
{
	// init effect params 
	viewDistance = 3;
	nearClipPlane = 1;
	farClipPlane = 30;
	fovAngle = 60.0;
	
	//Variablen zur Bildsequenzkontrolle
	Acc_curImg=0;
	Acc_lstLen=parentFrame->getNumImages(0);

	currentDepthCoord = 0.0;

	glClearColor(0.0,0.0,0.0,1);
	glShadeModel(GL_SMOOTH);

	glLoadIdentity();
	
	// set texturing parameters
	glEnable(GL_DEPTH_TEST);

	glEnable(GL_TEXTURE_2D);

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	
	glGenTextures(1, &texture);
	glBindTexture(GL_TEXTURE_2D, texture);

	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);
	
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST); 


}

/**
 * 
 * Erzeugt eine neues AccumEffectPanel Objekt.
 * Das neue AccumEffectPanel-Objekt bergibt einen Zeiger auf sein Panel an controlPanel.
 * Der effectTimer word initialisiert und connected.  
 *
 * @see GLEffect::createControlPanel() , AccumEffectPanel::getPanel()
 **/void AccumEffect::createControls(QWidget* parentWindow)
{
	cPan = new AccumEffectPanel(parentWindow); // Das Effekt-Panel
	controlPanel = cPan->getPanel();

	effectTimer = new QTimer(this, "none");
	connect(effectTimer, SIGNAL(timeout()), this, SLOT(animate()) );
	
}

/**
 * 
 * Startet die Animation (mit 50ms Bildverzgerung)
 *  
 **/void AccumEffect::play()
{
	effectTimer->start(50, false);
	
}

/**
 * 
 * Stoppt die Animation und setzt Acc_curImg=0 (also wieder auf das erste Bild zurck)
 *  
 **/
void AccumEffect::stop()
{
	effectTimer->stop();
		
	Acc_curImg=0;

}

/**
 * 
 * Setzt die Effektparameter zurck.
 * 
 * Setzt Acc_curImg=0 (also wieder auf das erste Bild zurck)
 * Setzt Acc_lstLen auf die Anzahl der Bilder der Sequenz.
 * 
 **/void AccumEffect::reset()
{
	effectTimer->stop();

	Acc_lstLen = parentFrame->getNumImages(0);
	Acc_curImg = 0;

	cPan->ResetAll(); // Setze alle Parameter des Effekt Panels auf den Urzustand zurck
	
	update();
}

/**
 * 
 * Pausiert die Animation.
 *  
 **/
void AccumEffect::pause()
{
	effectTimer->stop();
}

/**
 * 
 * Rendert den Effekt abhngig von der Effektauswahl 
 * welche AccumEffectPanel::getEffektAuswahl() liefert.
 *   
 * getEffektAuswahl() liefert 0
 * 
 *   Kein Effekt, die Sequenz wird normal abgespielt
 *
 * getEffektAuswahl() liefert 1
 *
 *   Motion Blur, Bewegungsschlieren. 
 *   Neues Bild=Altes Bild * fak + Neues Bild * (1 - Fak).
 *   Die Strke Fak dieses Effekts liefert AccumEffectPanel::getBlurFaktor()
 *   Zum Schluss wird noch das Ergebniss noch mit einer Gammakorrektur multipliziert.
 *   Den Wert dazu liefert die Funktion AccumEffectPanel::getGammaKorrektur().
 *  
 * getEffektAuswahl() liefert 2
 *
 * Bildsubtraktion. B# bedeutet Bild# (also B1 heit Bild1)
 *                    
 *   Neues Bild = y1(x1*B1 op1 x2*B2) + y2(x3*B3 op2 x4*B4) + y3(x5*B5 op3 x6*B6) 
 *      
 *   Die Rechenoperation op1,op2 und op3 (+ oder -) liefern die Funktionen 
 *    AccumEffectPanel::getOp1(),getOp2() und getOp3()
 * 
 *   Welche Bilder jeweils accumuliert werden sollen (B1-B6) liefern die Funktionen
 *    AccumEffectPanel::getBildAuswahl11(),getBildAuswahl12(),
 *                       getBildAuswahl21(),getBildAuswahl32(),
 *                       getBildAuswahl31(),getBildAuswahl32(),
 * 
 *   Die Intensitt der einzelenen Bilder (x1-x6) liefern die Funktionen
 *    AccumEffectPanel::getAnteilBild11(),getAnteilBild12(),
 *                      getAnteilBild21(),getAnteilBild22(),
 *                      getAnteilBild31(),getAnteilBild32(), 
 * 
 *   Die Helligkeitsanteile der einzelenen Paare (y1-y3) liefern die Funktionen
 *    AccumEffectPanel::getHellAnteil1(),getHellAnteil2(),getHellAnteil3()
 * 
 *   Zum Schluss wird noch das Ergebniss noch mit einer Gammakorrektur multipliziert.
 *    Den Wert dazu liefert die Funktion AccumEffectPanel::getGammaKorrektur()   
 **/
void AccumEffect::render()
{
	glEnable(GL_TEXTURE_2D);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

//*******************
//* Bildsubtraktion * 
//*******************
	if(cPan->getEffektAuswahl()==2)
	{
		int pa=cPan->getPaarAuswahl();
		int ha,ab,ba,ci;
			
		for(int z=0;z<(pa+1)*2;z++)
		{
			switch(z)		// ermittelt das als nchstes zu akkumulierende Bild
			{
				case 0:
					ba=cPan->getBildAuswahl11();
					if( !cPan->getFixFirst()) 
						ci=Acc_curImg;
					else ci=0;
					break;
				case 1:
					ba=cPan->getBildAuswahl12();			
					ci=Acc_curImg;
					break;
				case 2:
					ba=cPan->getBildAuswahl21();			
					if( !cPan->getFixFirst()) 
						ci=Acc_curImg;
					else ci=0;
					break;
				case 3:
					ba=cPan->getBildAuswahl22();			
					ci=Acc_curImg;
					break;
				case 4:
					ba=cPan->getBildAuswahl31();			
					if( !cPan->getFixFirst()) 
						ci=Acc_curImg;
					else ci=0;
					break;
				case 5:
					ba=cPan->getBildAuswahl32();			
					ci=Acc_curImg;
					break;
			}
			
			//Zeichnet ein Rechteck mit der Textur
			texImage1 = parentFrame->fetchImage( 0,( ci + ba ) % Acc_lstLen ); 
					
			if( texImage1 != 0 )
				glTexImage2D(GL_TEXTURE_2D,0,4,texImage1->width(), texImage1->height(),
					0, GL_RGBA, GL_UNSIGNED_BYTE, texImage1->bits());
			
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			glBindTexture( GL_TEXTURE_2D, texture );
			glScaled( 1.f , 1.f , 1.f );
			
			glBegin(GL_QUADS);
				glTexCoord2f(0,0);
				glVertex3f(-2,-1, 0);
				glTexCoord2f(0,1); 
				glVertex3f(-2, 1, 0);
				glTexCoord2f(1,1); 
				glVertex3f( 2, 1, 0);
				glTexCoord2f(1,0); 
				glVertex3f( 2,-1, 0);
			glEnd();
		
				
			switch(z) //ermittelt die Helligkeitsanteile der Bilder
			{
				case 0:
					ha=cPan->getHellAnteil1();
					ab=cPan->getAnteilBild11();
					glAccum(GL_LOAD,(GLfloat(ha)/100.f)*(GLfloat(ab)/100.f));
					// Das erste Bild berschreibt den Sccubuffer, die anderen werden
					// akkumuliert
					break;
				
				case 1:
					ha=cPan->getOp1()*cPan->getHellAnteil1();
					ab=cPan->getAnteilBild12();
					break;
					
				case 2:
					ha=cPan->getHellAnteil2();
					ab=cPan->getAnteilBild21();
					break;
					
				case 3:
					ha=cPan->getOp1()*cPan->getHellAnteil2();
					ab=cPan->getAnteilBild22();
					break;
					
				case 4:
					ha=cPan->getHellAnteil3();
					ab=cPan->getAnteilBild31();
					break;
				
				case 5:
					ha=cPan->getOp1()*cPan->getHellAnteil3();
					ab=cPan->getAnteilBild32();
					break;
			}
		
			if( z > 0 )		// Alle Bilder bis auf erste werden akkumuliert
				glAccum(GL_ACCUM, ( GLfloat(ha) / 100.f ) * ( GLfloat(ab) / 100.f ) );
			

		}
		
		// Zum Schluss noch die Gammakorrektur
		GLfloat gk = GLfloat( cPan->getGammaKorrektur() ) / 100.f;
		
		// Und das Bild wird wieder in den Framebuffer geschrieben
		glAccum(GL_RETURN,gk);
	}		

//**************
//* Motion Blur *
//**************
	if(cPan->getEffektAuswahl()==1)
	{
		//Zeichnet ein Rechteck mit der Textur
		texImage1 = parentFrame->fetchImage(0,Acc_curImg);

		if(texImage1!=0)
			glTexImage2D(GL_TEXTURE_2D,0,4,texImage1->width(), texImage1->height(),0, GL_RGBA, GL_UNSIGNED_BYTE, texImage1->bits());
			
		glBindTexture(GL_TEXTURE_2D, texture);
		glScaled(1.f,1.f, 1.f);
		glBegin(GL_QUADS);
			glTexCoord2f(0,0);
			glVertex3f(-2,-1, 0);
			glTexCoord2f(0,1);
			glVertex3f(-2, 1, 0);
			glTexCoord2f(1,1); 
			glVertex3f( 2, 1, 0);
			glTexCoord2f(1,0); 
			glVertex3f( 2,-1, 0);
		glEnd();

		// ermittelt die Blur-Verhltnisse
		int bf=cPan->getBlurFaktor();
		int gamma=cPan->getGammaKorrektur();
			
		GLfloat fak=(((GLfloat)bf)/100.f);
		GLfloat fak2=(((GLfloat)gamma)/100.f);
		
		// Altes Bild * fak + Neues Bild * (1 - Fak)
		glAccum(GL_MULT, fak);
		glAccum(GL_ACCUM, 1.0f-fak);
		
		// Schreibt das Resultat mit Faktor fak2 in den Framebuffer
		glAccum(GL_RETURN, fak2);
	}

//****************			
//* Kein Effekt	 *
//****************
	if(cPan->getEffektAuswahl()==0)
	{
		// Zeichnet einbfach nur ein Rechteck mit der Textur
		texImage1 = parentFrame->fetchImage(0,Acc_curImg);

		if(texImage1!=0)
			glTexImage2D(GL_TEXTURE_2D,0,4,texImage1->width(), texImage1->height(),
				0, GL_RGBA, GL_UNSIGNED_BYTE, texImage1->bits());
			
		glBindTexture(GL_TEXTURE_2D, texture);
		glScaled(1.f,1.f, 1.f);
		
		glBegin(GL_QUADS);
			glTexCoord2f(0,0);
			glVertex3f(-2,-1, 0);
			glTexCoord2f(0,1);
			glVertex3f(-2, 1, 0);
			glTexCoord2f(1,1); 
			glVertex3f( 2, 1, 0);
			glTexCoord2f(1,0);
			glVertex3f( 2,-1, 0);
		glEnd();
	}
}

/**
 * Steuert die Filmsequenz. Wird vom effectTimer Object aufgerufen (alle 50ms)
 * Wenn die Sequenz zu Ende ist, fngt sie wieder vorne an.
 *
 **/
void AccumEffect::animate()
{
	Acc_curImg++; //nchstes Bild in der Sequenz

	if(Acc_curImg>Acc_lstLen-1) // Wenn am Ende, dann fange wieder vorne an (Loop)
		Acc_curImg=0;

	update();
}

