
// system library headers
#include <stdlib.h>
#include <math.h>
#include <time.h>

#ifndef WIN32
#	include <unistd.h>
#	include <sys/times.h>
#endif

// Qt headers
#include <qimage.h>
#include <qtimer.h>

#include <qvgroupbox.h>
#include <qlabel.h>
#include <qslider.h>
#include <qcheckbox.h>

//local headers
#include "SampleEffect2.h"

const double PI = 3.14159;

SampleEffect2::SampleEffect2(GLFramework* pr)
	:	GLEffect(pr)
{

	//provide some basic info about the effect
	effectInfo->effectName = "Random jitter";
	effectInfo->version = "1.0";
	effectInfo->versionMajor = 1;
	effectInfo->shortDescription = "Distorts the grid in a random way...";
	effectInfo->author  ="Abe";

	effectInfo->requiredImageLists = 1;
	effectInfo->needNewTextures = true;
	effectInfo->needStreaming = false;
	effectInfo->needRawTextures = false;

	effectInfo->fileDialogNames[0]= "Load Textures";
	
	createControlPanel(false);

	texImage1 = 0;
	texImage2 = 0;
	texImage3 = 0;

	/* We want to build a 10x10 rectangular grid  here...*/
	gridHeight = 10; gridWidth = 10;

    maxDistortionOffset = 0.001F;
	cellSize = 0.1F;

	frameRate = 50;
	zDistort = false;
	explosionMode = false;


	// vertices are 3d floating point coords
	vertexArray = new GLfloat[gridWidth*gridHeight][3];

	// tex coords are 2d
	texCoordArray = new GLfloat[gridWidth*gridHeight][2];

	// color values are 3d
	colorArray = new GLfloat[gridWidth*gridHeight][3];

	// indices are unsigned integers
	// the index array will contain indices for the current triangle strip
	indexArray = new GLuint[2*gridWidth];

	hideControls();
	
}

SampleEffect2::~SampleEffect2()
{
	delete[] vertexArray;
	delete[] colorArray;
	delete[] indexArray;

	delete[] texCoordArray;
}



void SampleEffect2::createControls(QWidget* parentWindow)
{
	
	controlPanel = new QVGroupBox("Random Distortion",parentWindow,"test");
		
	//timer to enable animation in this effect
	effectTimer = new QTimer(this, "none");
	
	//a quick GUI 

	QLabel* dist_label = new QLabel("Distortion strength:", controlPanel);

	distSlider = new QSlider(1,100,2,2, QSlider::Horizontal, 												controlPanel, "test_slider");
	distSlider->setTickmarks(QSlider::Above);
	distSlider->setTickInterval(5);

	QLabel* rate_label = new QLabel("Animation speed:", controlPanel);

	rateSlider = new QSlider(50,250, 2,2, QSlider::Horizontal, 
				 controlPanel, "test_slider");
	rateSlider->setTickmarks(QSlider::Above);
	rateSlider->setTickInterval(5);

	zBox = new QCheckBox("Distort Z coordinate",controlPanel);
	explodeBox = new QCheckBox("Wacky mode", controlPanel);
	pointBox = new QCheckBox("Show vertices", controlPanel);
	lineBox = new QCheckBox("Show grid lines", controlPanel);
	textureBox = new QCheckBox("Show texture", controlPanel);

	//connect controls to the effect parameters
	

	connect(effectTimer, SIGNAL(timeout()), this, SLOT(animate()) );

	connect(zBox, SIGNAL(toggled(bool)), 
		this, SLOT(setZDistortion(bool)) );
	
	connect(explodeBox, SIGNAL(toggled(bool)),
		this, SLOT(setExplosionMode(bool)) );
	
	connect(pointBox, SIGNAL(toggled(bool)),
		this, SLOT(setPointMode(bool)) );
	
	connect(lineBox, SIGNAL(toggled(bool)),
		this, SLOT(setLineMode(bool)) );
	
	connect(textureBox, SIGNAL(toggled(bool)),
		this, SLOT(setTextureMode(bool)) );


	connect(rateSlider, SIGNAL(valueChanged(int)), 
		this, SLOT(setFrameRate(int)) );
	
	connect(distSlider, SIGNAL(valueChanged(int)), 
		this, SLOT(setDistortionStrength(int)) );

}

// slots

void SampleEffect2::setDistortionStrength(int strength)
{
	maxDistortionOffset = 0.05*((GLdouble)strength/100.0);
	update();

}

void SampleEffect2::setZDistortion(bool enable)
{
	zDistort = enable;
	update();

}

void SampleEffect2::setFrameRate(int rate)
{
	
	effectTimer->changeInterval(300-rate);
	frameRate = rate;
	
	update();
}

void SampleEffect2::setExplosionMode(bool enable)
{
	explosionMode = enable;
	update();
}

void SampleEffect2::setPointMode(bool enable)
{
	showPoints = enable;
	update();
}

void SampleEffect2::setLineMode(bool enable)
{
	showLines = enable;
	update();
}

void SampleEffect2::setTextureMode(bool enable)
{
	showTexture = enable;
	update();
}

// control functions

void SampleEffect2::play()
{
	// start animation
	effectTimer->start(300-frameRate,false);

}

void SampleEffect2::stop()
{
	// stop and go back to the first frame
	effectTimer->stop();
	texImage1 = parentFrame->fetchImage(0,0);
	update();
}

void SampleEffect2::reset()
{
	// stop and reset effect
	effectTimer->stop();
	texImage1 = parentFrame->fetchImage(0,0);

	distSlider->setValue(1);
	rateSlider->setValue(50);

	zBox->setChecked(false);
	explodeBox->setChecked(false);
	lineBox->setChecked(false);
	pointBox->setChecked(false);
	textureBox->setChecked(true);

	zDistort = false;
	explosionMode = false;
	showPoints = false;
	showLines = false;
	showTexture = true;

	maxDistortionOffset = 0.001F;
	frameRate = 50;

	initGrid();
	update();

}

void SampleEffect2::pause()
{
  //pause animation
  effectTimer->stop();
}

void SampleEffect2::animate()
{
	for(GLuint i=0;i < gridHeight;i++)
	{
		for(GLuint j=0; j < gridWidth; j++)
		{
			//distort xyz coords of current vertex
			distortVertex(vertexArray[j+(i*gridWidth)][0],
					vertexArray[j+(i*gridWidth)][1], 
					vertexArray[j+(i*gridWidth)][2],
					j, i);
		}
	}
  // go to next image 
  texImage1 = parentFrame->fetchNextImage(0); 
  update();
}

//grid initialization

void SampleEffect2::initGrid()
{
	// initialize vertex, color, and tex coord. arrays
	//int row = 0; // row number
	GLfloat xcoord = -0.5; // coords of the upper-left coord of the grid
	GLfloat ycoord = -0.5;
	
	GLfloat xTexCoord = 0.0;
	GLfloat yTexCoord = 0.0;
	
	for(GLuint i=0;i < gridHeight;i++)
	{
		for(GLuint j=0; j < gridWidth; j++)
		{
			// x-coord
			vertexArray[j+(i*gridWidth)][0] = xcoord; 
		
			// y-coord
			vertexArray[j+(i*gridWidth)][1] = ycoord;
		
			//z-coord
			vertexArray[j+(i*gridWidth)][2] = 0; 
			//planar grid for now

			// set tex coords...
			texCoordArray[j+(i*gridWidth)][0] = xTexCoord;
			texCoordArray[j+(i*gridWidth)][1] = yTexCoord;

			// set color 
			colorArray[j+(i*gridWidth)][0] = 1.0; //(GLdouble)rand()/RAND_MAX;
			colorArray[j+(i*gridWidth)][1] = 0.0; //(GLdouble)rand()/RAND_MAX;
			colorArray[j+(i*gridWidth)][2] = 0.0; //(GLdouble)rand()/RAND_MAX;

			xcoord += 0.1F;
			xTexCoord += 0.1F;

		}
		xcoord = -0.5F; //reset xcoord 
		ycoord += 0.1F; //go the next row

		xTexCoord = 0.0F;
		yTexCoord += 0.1F;
	}

}

// distortion effect function

void SampleEffect2::distortVertex(GLfloat& xcoord, GLfloat& ycoord, 
				 GLfloat& zcoord, GLint xposition, 
				 GLint yposition)
{
  //calculate reference points for the vertex
  GLdouble xcoordPos = -0.5 + xposition*cellSize;
  GLdouble ycoordPos = -0.5 + yposition*cellSize;
  GLdouble zcoordPos = 0;

  xcoord += maxDistortionOffset * ( (GLdouble)rand() 
		  / ((GLdouble)RAND_MAX/2) -1.0 );
  ycoord += maxDistortionOffset * ( (GLdouble)rand()
		  / ((GLdouble)RAND_MAX/2) -1.0 );
  
  if(zDistort)
    zcoord += maxDistortionOffset * ( (GLdouble)rand() 
		    / ((GLdouble)RAND_MAX/2) -1.0 );


  //caculate distances from the distorted positon to the reference point
  GLdouble xdist = (xcoord-xcoordPos>0)?(xcoord-xcoordPos):-(xcoord-xcoordPos);
  
  GLdouble ydist = (ycoord-ycoordPos>0)?(ycoord-ycoordPos):-(ycoord-ycoordPos);

  GLdouble zdist = (zcoord-zcoordPos>0)?(zcoord-zcoordPos):-(zcoord-zcoordPos);

  if(explosionMode==false)
  {
	if(xdist > cellSize/2.0)
	  xcoord = xcoordPos;

	if(ydist > cellSize/2.0)
	  ycoord = ycoordPos;

	if(zdist > cellSize/2.0)
	  zcoord = zcoordPos;
  }

}

// initialization functions

void SampleEffect2::initialize()
{
	
	// initialize effect parameters and controls
	viewDistance = 5;
	nearClipPlane = 1;
	farClipPlane = 30;
	fovAngle  = 60;

	currentDepthCoord = 0.0;

	glClearColor(0.0,0.0,0.0,1);
	glShadeModel(GL_SMOOTH);

	glLoadIdentity();
	
	zDistort = false;
	explosionMode = false;
	showPoints = false;
	showLines = false;
	showTexture = true;

	textureBox->setChecked(true);

	maxDistortionOffset = 0.001F;
	frameRate = 50;

	// set texturing parameters
	glEnable(GL_DEPTH_TEST);

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	
	glGenTextures(1, &texture);
	glBindTexture(GL_TEXTURE_2D, texture);

	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);
	
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_LINEAR);

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST); 
	
	initGrid();

	texImage1 = parentFrame->fetchNextImage(0);

}	

void SampleEffect2::render()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	glEnableClientState(GL_COLOR_ARRAY);
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	
	glVertexPointer(3,GL_FLOAT, 0, vertexArray);
	glColorPointer(3, GL_FLOAT, 0, colorArray);
	glTexCoordPointer(2,GL_FLOAT, 0, texCoordArray);

	glEnable(GL_TEXTURE_2D);

	if(texImage1!=0)
	glTexImage2D(GL_TEXTURE_2D,0,4,texImage1->width(), texImage1->height(),
			0, GL_RGBA, GL_UNSIGNED_BYTE, texImage1->bits());
	
	glBindTexture(GL_TEXTURE_2D, texture);

	glScaled(2.0, 2.0, 2.0);
	
	int offset = gridWidth;
	int BaseIndex = 0;

	glPointSize(3.0);
	
	//draw the textured grid 
	for (GLuint y=0; y<gridHeight-1; y++)
	{
		for (GLuint x = 0 ; x<gridWidth ; x++)
		{
			indexArray[x<<1] = BaseIndex+x;
			indexArray[(x<<1)+1] = BaseIndex+x+offset;
		}
		
		if(showTexture)
		{
			//draw textured grid
			glEnable(GL_TEXTURE_2D);
			glDrawElements(GL_TRIANGLE_STRIP,2*gridWidth,
					GL_UNSIGNED_INT,indexArray);
		}

		if(showPoints)
		{
		  //show grid vertices
		  glDisable(GL_TEXTURE_2D);
		  glDrawElements(GL_POINTS,(gridWidth)*2,
				  GL_UNSIGNED_INT,indexArray);
		 
		}
		
		if(showLines)
		{
		  //show grid lines
		  glDisable(GL_TEXTURE_2D);
		  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
		  glDrawElements(GL_TRIANGLE_STRIP,2*gridWidth,
				  GL_UNSIGNED_INT,indexArray);
		  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		  
		}
		
		// go to next row
		BaseIndex += gridWidth;
	} 
	
	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	glDisable(GL_TEXTURE_2D);

}



