
// Qt headers
#include <qgl.h>

#include <qtimer.h>
#include <qvgroupbox.h>
#include <qlabel.h>
#include <qslider.h>

// system library headers
#include <stdlib.h>
#include <math.h>
#include <iostream.h>

//local headers
#include "SampleEffect.h"

const double PI = 3.14159;
const unsigned int UNDEFINED = 0xFFFFFFFF;

SampleEffect::SampleEffect( GLFramework* pr)
	:	GLEffect(pr)
{

	//provide some basic info about the effect
	effectInfo->effectName="Sample Effect #1";
	effectInfo->version="1.0";
	effectInfo->versionMajor=1;
	effectInfo->shortDescription="Parametric polygon figures";
	effectInfo->author="Abe";
	effectInfo->requiredImageLists=0;
	effectInfo->needNewTextures=false;
	effectInfo->needStreaming=false;

	createControlPanel(true);

	//temporary 

}

SampleEffect::~SampleEffect()
{
}



void SampleEffect::createControls(QWidget* parentWindow)
{
	// create the control panel
	controlPanel = new QVGroupBox("Parametric stars",parentWindow,"stars");
		
	// timer to enable animation in this effect
	effectTimer = new QTimer(this, "none");
	
	// a quick GUI - this code is wayyy dirty yet...
	
	vertex_label = new QLabel("Number of vertices:", controlPanel);
	vertex_slider = new QSlider(2,30,2,2, QSlider::Horizontal, 
						controlPanel, "test_slider");
	
	vertex_slider->setTickmarks(QSlider::Above);
	vertex_slider->setTickInterval(2);
	

	radius_label = new QLabel("Outter radius:", controlPanel);
	radius_slider = new QSlider(5,20,1,1, QSlider::Horizontal, 
						controlPanel, "test_slider");
	
	radius_slider->setTickmarks(QSlider::Above);
	radius_slider->setTickInterval(1);

	inner_radius_label = new QLabel("Inner radius:", controlPanel);
	inner_radius_slider = new QSlider(20,100,1,1, QSlider::Horizontal, 
							controlPanel, "test_slider");
	
	inner_radius_slider->setTickmarks(QSlider::Above);
	inner_radius_slider->setTickInterval(5);


	angle_label = new QLabel("Angular velocity:", controlPanel);
	angle_slider = new QSlider(1,45,1,1, QSlider::Horizontal, 
						controlPanel, "test_slider");
	
	angle_slider->setTickmarks(QSlider::Above);
	angle_slider->setTickInterval(5);

	//connect controls to the effect parameters

	connect(vertex_slider, SIGNAL(valueChanged(int)), 
			this, SLOT(setNumVertices(int)) );
	connect(radius_slider, SIGNAL(valueChanged(int)), 
			this, SLOT(setRadius(int)) );
	
	connect(inner_radius_slider, SIGNAL(valueChanged(int)), 
			this, SLOT(setInnerRadius(int)) );
	
	connect(angle_slider, SIGNAL(valueChanged(int)), 
			this, SLOT(setAngularVelocity(int)) );
	
	connect(effectTimer, SIGNAL(timeout()), 
			this, SLOT(animate()) );
	
	//set initial effect parameters

	numVertices=4;
	radius=0.5;
	innerRadius=0.2*radius;
	rotationAngle=.5;
	
}

// test wheel event handler
void SampleEffect::wheelEvent(QWheelEvent* event)
{
	currentHue += (event->delta())/(120);
	
	HLS_to_RGB( &starColor[0], &starColor[1], &starColor[2],
				 currentHue, 0.6f, 1.0f);
	update();
}

// slots

void SampleEffect::setAngularVelocity(int ang)
{
	rotationAngle=ang; update();
}

// change number of vertices
void SampleEffect::setNumVertices(int vertices)
{
	numVertices=vertices*2; update();

}

void SampleEffect::setRadius(int rad)
{
	//normalize radius to [0...1]
	radius = rad/20.0; update();
}

void SampleEffect::setInnerRadius(int inr)
{
  // inner radius is in % of the outter radius
  innerRadius = (radius/100.0)*((inr));
  update();
}


// control functions

void SampleEffect::play()
{
	effectTimer->start(10,false);

}

void SampleEffect::stop()
{
	effectTimer->stop();

}

void SampleEffect::reset()
{
	stop();

	angle_slider->setValue(0);
	radius_slider->setValue(5);
	vertex_slider->setValue(4);
	inner_radius_slider->setValue(20);

	update();
	
}

void SampleEffect::pause()
{
	effectTimer->stop();
}

void SampleEffect::animate()
{
	currentAngle += rotationAngle;
	if(currentAngle>360.0f) currentAngle = 0.0f;

	update();
}

// initialization functions

void SampleEffect::initialize()
{
	glClearColor(0.0,0.0,0.0,1);
	starColor[0]=starColor[1]=starColor[2]=1.0f;
	
	currentHue=0.0f;

	fovAngle = 60;
	nearClipPlane = 1;
	farClipPlane = 30;
	viewDistance = 3;

	currentAngle = 0;
	currentDepthCoord = 0;

}


/* the sample effect routine is here
	This simply builds symmetrical n-gons which can 
	optonially be turned into a star
 */
void SampleEffect::render()
{	
	//starColor[0]=starColor[1]=starColor[2]=1.0F;
	
	glClear(GL_COLOR_BUFFER_BIT);

	GLfloat angle_incr=(2*PI)/(float)numVertices;
	GLfloat angle=0.0;

	// rotate the scene
	glRotated(-currentAngle,0,0,1);	

	//glColor3f(0.0F,0.7F,1.0F);
	glColor3fv(starColor);

	// draw the star 
	glBegin(GL_LINE_LOOP);
	for(int i=0;i<numVertices;i++)
	{
	
		if((i)%2)
		{
			glVertex3f(radius*cos(angle), radius*sin(angle),0.0f);
		}
		else
		{
			
			glVertex3f(innerRadius*cos(angle+angle_incr/2.0), 
				   innerRadius*sin(angle+angle_incr/2.0),0.0f);
		}
		
		angle+=angle_incr;
	}
	
	glEnd();

}

// helper functions

void SampleEffect::HLS_to_RGB(float *r, float *g, float *b, float h, float l, float s)
{
 // Given: h in [0, 360] or UNDEFINED, l and s in [0,1]
 // Desired: r,g,b each in [0,1]

	float min, max, f; 
	double dummy;

	if(s==0.0) // Achromatic case
	{
	  if(h == UNDEFINED)
	  {
		*r = *g = *b = l; 
		return;
	  }

	  else { cerr<<"Hue !=0 for achromatic case!";exit(-1);}

	}

	// normalize hue to 0...360
	if(h>360.0) h-=360.0;
	if(h<0.0) h+=360.0;

	if(l<=0.5)
	{
		max = l*(1+s);
	}
	else max = l+s-l*s;

	min = 2*l-max;

	f = modf(h/360.0, &dummy);

	// for the sectors in the vicinity of 'primary'hues
	
	// 'red' sector
	if((h>=330.0) | (h<=30.0))
	{
		*r = max;
		*g = min + f*(max-min);
		*b = min;
	}

	// 'green' sector
	if(((h>=90.0) & (h<=150.0)))
	{
		*r = min;
		*g = max;
		*b = min + f*(max-min);
	}
	
	// 'blue' sector
	if(((h>=210.0) & (h<=270.0)))
	{
		*r = min + f*(max-min);
		*g = min;
		*b = max;
	}

	// for sectors in between the primary hues
	
	// 'yellow' sector
	if((h<90.0) & (h>30.0))
	{
		*r = min + (1-f)*(max-min);
		*g = max;
		*b = min;
	}

	// 'cyan' sector
	if(((h>150.0) & (h<210.0)))
	{
		*r = min;
		*g = min + (1-f)*(max-min);
		*b = max;
	}
	
	// 'magenta' sector
	if(((h>270.0) & (h<330.0)))
	{
		*r = max;
		*g = min;
		*b = min + (1-f)*(max-min);
	}

}

	


